(ns jaq.core
  (:gen-class)
  (:require
   [bidi.ring :refer (make-handler)]
   [clojure.tools.logging :as log]
   [ring.middleware.defaults :refer [wrap-defaults site-defaults]]
   [ring.middleware.nested-params :as nested-params]
   [ring.middleware.keyword-params :as keyword-params]
   [ring.middleware.session :as session]
   [ring.middleware.params :as params]
   [ring.middleware.content-type :refer [wrap-content-type]]
   [ring.middleware.not-modified :refer [wrap-not-modified]]
   [ring.middleware.format :refer [wrap-restful-format]]
   [ring.middleware.file :refer [wrap-file]]
   [ring.middleware.resource :refer [wrap-resource]]
   [ring.util.servlet :refer [defservice]]
   [ring.util.response :as res]
   [hiccup.page :as hiccup]
   [jaq.services.storage :as storage]
   [jaq.services.appengine-admin :as admin]
   [jaq.services.resource :as resource]
   [jaq.services.deferred :refer [with-deferred] :as deferred]
   [jaq.services.datastore :as datastore]
   [jaq.services.memcache :as memcache]
   [jaq.services.management :as management]
   [jaq.services.resource :as resource]
   [jaq.services.auth :as auth]
   [jaq.services.util :refer [dev? repl-server remote!]]))


;;;;;;;; Request handler

(defmulti api-fn
  "Dispatcher for client api calls"
  :fn)

(defmethod api-fn :reload [params]
  (memcache/pop [:reload]))

(defmethod api-fn :syncer [{:keys [id messages]}]
  (let [tasks (deferred/lease {:tag id})
        out (deferred/process tasks)
        _ (-> (deferred/delete tasks)
              (doall))
        _ (->> messages
               (map deferred/defer)
               (doall))]
    {:fn :syncer :id id :messages out}))

;;;;;;;; Messages
(def session-entity :core/sessions)
(def session-id 1)
(def session-key (str session-entity "/" session-id))
(defonce session-store (datastore/create-store session-entity))

(defn get-sessions [key]
  (let [m (memcache/peek key)
        sessions (get m key)]
    (if sessions
      sessions
      (let [session-entity {:id session-id :v 1 :devices []}
            sessions (try
                       (datastore/fetch session-store session-id 1)
                       (catch Exception e
                         (datastore/store-all! session-store [session-entity])
                         session-entity))]
        (memcache/push {key sessions})
        sessions))))

(defn bootstrap []
  (get-sessions session-key))

(defmethod deferred/defer-fn :open-session [{:keys [device-id]}]
  (let [sessions (get-sessions session-key)
        devices (:devices sessions)]
    (when-not (contains? (set devices) device-id)
      (memcache/push {session-key
                      (datastore/update! session-store session-id update :devices conj device-id)}))))

(defmethod deferred/defer-fn :close-session [{:keys [device-id]}]
  (let [sessions (datastore/update! session-store session-id update :devices (partial remove #(= device-id %)))]
    (memcache/push {session-key sessions})))

(def one-day-ms (* 1000 60 60 24))

(defmethod deferred/defer-fn :expire-tasks [{:keys [expiration-ms]
                                             :or {expiration-ms one-day-ms}}]
  (-> (doseq [task (deferred/lease)
              :let [eta (.getEtaMillis task)
                    age (- (System/currentTimeMillis) eta)]
              :when (> age expiration-ms)]
        task)
      (deferred/delete)
      (doall)))

(defmethod deferred/defer-fn :broadcast-message [{:keys [value timestamp device-id]
                                                  :as m}]
  (let [{:keys [devices]} (get-sessions session-key)]
    (log/info :broadcast-message m devices)
    (->> devices
         (remove #(= device-id %))
         (map #(deferred/add (merge m {:fn :new-message}) %))
         (doall))))

;;;;; Default
(defmethod api-fn :default [params]
  {:error "Unknown fn call"
   :params params})

(defn api-handler [request]
  (let [params (:body-params request)]
    (res/response (api-fn params))))

;;;; HTML pages
(def html-content-type "text/html")
(def bootstrap-css "https://maxcdn.bootstrapcdn.com/bootstrap/4.0.0-beta.2/css/bootstrap.min.css")

(defn as-html [html]
  (res/content-type
   (res/response html)
   html-content-type))

(defn index-handler [request]
  (as-html
   (hiccup/html5
    [:head
     [:meta {:charset "utf-8"}]
     [:meta {:name "viewport" :content "width=device-width, initial-scale=1.0, maximum-scale=3.0, user-scalable=yes"}]
     [:title "Jaq"]
     [:link {:href "/static/images/favicon.ico" :rel "shortcut icon" :type "image/x-icon"}]
     (hiccup/include-css bootstrap-css)
     #_(hiccup/include-css "/static/css/main.css")
     [:style {:type "text/css"} ""]]
    [:body
     [:div#app.content.row.col
      [:p "Hello world!"]]
     #_(hiccup/include-js (str "/static/js/app.js"))])))

;;;; Routes and Handlers

(def routes ["" [["/" :index]
                 ["/api" :api]]])

(def handlers {:index #'index-handler
               :api #'api-handler})

;;;; Lifecycle handlers

(defn init []
  (log/info "starting")
  ;;(bootstrap)
  (when dev?
    (log/info "Starting REPL server")
    (repl-server)))

(defn destroy []
  (log/info "shutting down"))

;;;; Ring middleware config

(defn config []
  (-> site-defaults
      (assoc-in [:security :anti-forgery] false)
      (assoc-in [:security :xss-protection :enable?] false)))

(def app
  (->
   (wrap-defaults (make-handler routes #'handlers) (config))
   (wrap-restful-format)
   ((fn [e]
      (if dev?
        (wrap-file e "resources/public")
        (wrap-resource e "public"))))
   (wrap-content-type)
   (wrap-not-modified)))

(defservice app)
