(ns top9.app
  (:require [clojure.browser.repl :as repl]
            [clojure.string :as string]
            [cljs.reader :as edn]
            [cljs.core.async :refer [put! chan <! >! timeout close! onto-chan]]
            [cljs-time.core :as time-core]
            [cljs-time.coerce :as time-coerce]
            [cljs-time.format :as time-format]
            [goog.net.jsloader :refer [safeLoad]]
            [goog.html.TrustedResourceUrl :refer [format]]
            [goog.string.Const :refer [from]]
            [goog.object :as gobj]
            [taoensso.timbre :as log]
            [reagent.core :as r]
            [reagent.cookies :as cookies]
            [diffit.map :as diffit]
            [ajax.core :refer [GET POST]])
  (:require-macros
   [cljs.core.async.macros :refer [go go-loop]]
   [cljs.core :refer [goog-define]])
  (:import [goog]
           [goog.async Deferred]
           [goog.net.cookies]))

(goog-define env-prod "development")

(def prod? (= "production" env-prod))
(def dev? (not prod?))

(when dev?
  (enable-console-print!)

  (def base-url "/static/js/out")

  (defn page-uri []
    (goog.Uri. (.. js/window -location -href)))

  (defn normalize-href-or-uri [href-or-uri]
    (let [uri  (goog.Uri. href-or-uri)]
      (.getPath (.resolve (page-uri) uri))))

  (def this-ns
    (apply str (drop-last 2 (str `_))))

  #_(defn this-source [ns]
      (str base-url "/"
           (->> (string/split ns ".")
                (string/join "/"))
           ".js"))

  (defn path [ns]
    (goog.object/get js/goog.dependencies_.nameToPath ns))

  #_(defn do-update []
      (-> this-ns this-source from format safeLoad)))

;;;;;

(defn get-device-id []
  (if (cookies/contains-key? :device-id)
    (cookies/get :device-id)
    (let [device-id (str (random-uuid))]
      (cookies/set! :device-id device-id)
      device-id)))

;;;;; State ;;;;;

(defonce state (r/atom {:init false
                        :device-id (get-device-id)
                        :session-id nil}))

;;;;
(defn get-session-id []
  (let [session-id (get :session-id @state (str (random-uuid)))]
    (when-not (= session-id (:session-id @state))
      (swap! state assoc :session-id session-id))
    session-id))

;;;; Message handling ;;;;
(def polling 5)

(defonce control-ch (chan))
(defonce in-ch (chan))
(defonce event-ch (chan))

(defonce message-outbox (atom []))

;;;;; helpers ;;;;;;

(defn handler [k]
  (fn [e]
    (swap! state assoc-in [k] (-> e .-target .-value))))

(defn fire! [fn k & [r]]
  (put! event-ch {:fn fn
                  :value (get-in @state k)})
  (swap! state assoc-in k r))

;;;;; Components ;;;;;

(defn card [& [{:keys [header title sub-title text]}]]
  [:div.card.m-2
   (when-not (empty? header)
     [:div.card-header header])
   [:div.card-body.m-2
    (when-not (empty? title)
      [:h4.card-title.mb-2 title])
    (when-not (empty? sub-title)
      [:h6.card-subtitle.text-muted sub-title])
    (when-not (empty? text)
      [:div.card-text text])]])

(defn app []
  [:div.container
   [:div.row
    [:div.col.mt-5]]
   [:div.row
    [:div.col
     (map (fn [{:keys [value timestamp device-id]}]
            [:div {:key timestamp}
             (card {:header (str "Message" " by " device-id)
                    :text value})]) (->> (:messages @state)
                                   (take 5)
                                   (reverse)))]]
   [:div.row
    [:div.col.mt-5]]
   [:div.row
    [:div.col
     (card {:header "Your Message"
            :text [:div
                   [:input.form-control {:type :text
                                         :on-change (handler ::message)
                                         :value (::message @state)
                                         :on-key-press (fn [e]
                                                         (when (and (not (empty? (::message @state))) (= "Enter" (.-key e)))
                                                           (fire! ::add-message [::message] "")))}]]})]]
   [:div.row
    [:div.col.mt-5]]
   [:div.row
    [:div.col
     [:div
      (card {:header "Info" :title "A Title" :sub-title "The more you know"
             :text [:div
                    [:p "Your ID: " (get-in @state [:device-id])]
                    [:p "State: " (pr-str @state)]]})]]]])

;;;;;; Utils ;;;;;;

(defn enqueue [m]
  (swap! message-outbox conj m))

(defn current-timestamp
  "Current timestamp in UTC."
  []
  (-> (time-core/now)
      (time-coerce/to-long)))

;;;;;

(defmulti api-fn :fn)

(defmethod api-fn ::add-message [{:keys [value timestamp device-id]
                                  :or {timestamp (current-timestamp)
                                       device-id (:device-id @state)}
                                  :as m}]
  (let [message {:value value :timestamp timestamp :device-id device-id}]
    (swap! state update-in [:messages] conj message)
    (enqueue (merge message {:fn :broadcast-message}))))

(defmethod api-fn :new-message [{:keys [value timestamp device-id]
                                 :as m}]
  (log/info :new-message m)
  (swap! state update-in [:messages] conj m))

(defmethod api-fn :pong [{:keys [id cnt]
                          :or {cnt 0}}]
  (enqueue {:fn :ping :id id :cnt (inc cnt)})
  (swap! state assoc-in [:output] cnt))

(defmethod api-fn :default [{:keys [fn]}]
  (log/error "Unknown fn:" fn))

;;;;
(defn message-processor
  "Process incoming messages."
  [in-ch process-fn control-ch]
  (go-loop []
    (let [[e ch] (alts! [control-ch in-ch])]
      (when-not (= ch control-ch)
        (process-fn e)
        (recur)))))

;; an event: [:entity :action :context :timestamp]

(defn mailman
  "Poor man's bi-directional communication."
  [id in-ch outbox control-ch polling-ms]
  (go-loop []
    (let [[_ ch] (alts! [control-ch (timeout polling-ms)])
          out @outbox
          cnt (count out)]
      (when-not (= ch control-ch)
        (POST "/api" {:format :transit
                      :response-format :transit
                      :keywords? true
                      :params {:fn :syncer :id id :messages out}
                      :handler (fn [{:keys [id messages] :as m}]
                                 (swap! outbox (fn [e] (drop cnt e)))
                                 (onto-chan in-ch messages false))
                      :error-handler (fn [e]
                                       (log/warn "Transient network error:" e))})
        (recur)))))

(defn start! []
  (let [device-id (get-in @state [:device-id])]
    (log/debug "Starting message processor incoming")
    (message-processor in-ch api-fn control-ch)
    (log/debug "Starting message processor events")
    (message-processor event-ch api-fn control-ch)
    (log/debug "Starting mailman")
    (mailman device-id in-ch message-outbox control-ch (* 1000 polling))
    (log/debug "Starting session")
    (enqueue {:fn :open-session :device-id device-id})))

;;;;

(defn reload []
  (r/render [app] (.getElementById js/document "app")))

(defn ^:export main []
  (swap! state assoc :init true)
  (reload)
  (when prod?
    (start!))
  (when dev?
    (log/info "Starting REPL")
    (repl/connect "http://localhost:3449/repl")))

(defn ^:export exit []
  (enqueue {:fn :close-session :device-id (get-device-id)})
  ;;TODO drain channel before exiting
  nil)

;; https://stackoverflow.com/a/35734141/7947020
;;(set! js/cljs-entry-point main)
(when-not (:init @state)
  (set! (.-onload js/window) main)
  (set! (.-beforeunload js/window) exit))
