(ns ring-date.core
  "a date middleware, might expand to include more, be more helpful."
  {:author "Casey Mattingly (@akacase)"}
  (:require
   [clj-time.coerce :as tc]))

;; iso8601 and rfc3339 are very similar, this should be a catch all
;; for them
(def ^:private iso8601-re (re-pattern "^\\d{4}-\\d{2}-\\d{2}.*"))

(defn- date? [date-str]                                                         
  (when (and date-str (string? date-str))
    (re-matches iso8601-re date-str)))

(defn- json->datetime [json-str]
  "conversion function, converts `json-str` into joda instance, this
function doesn't throw an exception, it relies on spec to ensure
everything is correct"
  (when (date? json-str)
    (if-let [res (tc/from-string json-str)]                                     
      res
      nil))) 

(defn dt [m]
  "handler function to walk the request map and transform relevant
fields into joda instances"
  (let [f (fn [[k v]]
            (if (date? v)
              [k (json->datetime v)]                                           
              [k v]))]
    (clojure.walk/postwalk (fn [x] (if (map? x) (into {} (map f x)) x)) m)))

(defn wrap-date [handler]                                     
  "custom handler to serialize date fields into joda instances"
  (fn [req]
    (handler (update-in req [:body] #(dt %)))))
