(ns people-shared.utils
  (:require [clojure.string :as string]
            [clojure.walk :as walk]
            [clj-time.core :as t]
            [clj-time.coerce :as c]
            [clj-time.format :as f]))

(defn compact [x]
  (cond
   (vector? x) (into [] (filter (complement nil?) (map compact x)))
   (map? x) (into {} (filter (comp not nil? second) (map compact x)))
   :else x))

(defn ->uuid [x]
  (if (instance? java.util.UUID x)
    x
    (java.util.UUID/fromString x)))

(defn parse-timestamp [s]
  (when s
    (c/to-date
     (f/parse
      (f/formatters :date-hour-minute-second-ms)
      (string/replace s #"Z$" "")))))

(defn parse-date-time [s]
  (f/parse (f/formatters :date-time) s))

(defn parse-db-uri [s]
  (let [uri (java.net.URI. s)
        host (.getHost uri)
        port (if (pos? (.getPort uri)) (.getPort uri))
        path (.getPath uri)
        scheme (.getScheme uri)
        ^String query (.getQuery uri)
        query-parts (and query (for [^String kvs (.split query "&")]
                                 (vec (.split kvs "="))))]
    (merge
     {:subname (if host
                 (if port
                   (str "//" host ":" port path)
                   (str "//" host path))
                 (.getSchemeSpecificPart uri))
      :classname "org.postgresql.Driver"
      :subprotocol scheme }
     (if-let [user-info (.getUserInfo uri)]
       {:user (first (string/split user-info #":"))
        :password (second (string/split user-info #":"))})
     (walk/keywordize-keys (into {} query-parts)))))
