(ns com.adgoji.creatives.core
  (:require
   [com.adgoji.displayvideo-spec.interface :as displayvideo-spec]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.sdk-client-utils.interface :as sdk-client-utils]
   [com.adgoji.spec.interface :as spec])
  (:import
   (com.google.api.services.displayvideo.v4 DisplayVideo DisplayVideo$Advertisers$Creatives$List)
   (com.google.api.services.displayvideo.v4.model Creative)))

(defn create
  [^DisplayVideo client advertiser-id creative]
  (let [creative-conformed (spec/check creative ::displayvideo-spec/creative)
        creative-request   (java-utils/clj->java Creative creative-conformed)]
    (-> client
        (.advertisers)
        (.creatives)
        (.create (long advertiser-id) creative-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo-spec/creative))))

(defn delete
  [^DisplayVideo client advertiser-id creative-id]
  (-> client
      (.advertisers)
      (.creatives)
      (.delete (long advertiser-id) (long creative-id))
      (sdk-client-utils/execute-and-convert)))

(defn get-by-id
  [^DisplayVideo client advertiser-id creative-id]
  (-> client
      (.advertisers)
      (.creatives)
      (.get (long advertiser-id) (long creative-id))
      (sdk-client-utils/execute-and-convert)
      (spec/check ::displayvideo-spec/creative)))

(defn- fetch-creatives-page
  [^DisplayVideo$Advertisers$Creatives$List request ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (sdk-client-utils/execute-and-convert)))

(defn get-list
  [^DisplayVideo client advertiser-id {:keys [order-by filter-expr limit]}]
  (let [request (-> client
                    (.advertisers)
                    (.creatives)
                    (.list (long advertiser-id)))]
    (when order-by
      (.setOrderBy ^DisplayVideo$Advertisers$Creatives$List request order-by))
    (when filter-expr
      (.setFilter ^DisplayVideo$Advertisers$Creatives$List request filter-expr))
    (-> (sdk-client-utils/execute-and-convert-paginated (partial fetch-creatives-page request)
                                                        :creatives
                                                        limit)
        (spec/check ::displayvideo-spec/creatives))))

(defn update-by-id
  [^DisplayVideo client advertiser-id creative-id creative update-mask]
  (let [mask             (-> update-mask
                             (spec/check ::displayvideo-spec/update-mask)
                             sdk-client-utils/normalize-mask)
        creative-request (java-utils/clj->java Creative creative)]
    (-> client
        (.advertisers)
        (.creatives)
        (.patch (long advertiser-id) (long creative-id) creative-request)
        (.setUpdateMask mask)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo-spec/creative))))
