(ns com.adgoji.partner-channels.core
  (:require
   [com.adgoji.displayvideo-spec.interface :as displayvideo]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.sdk-client-utils.interface :as sdk-client-utils]
   [com.adgoji.spec.interface :as spec]
   [camel-snake-kebab.core :as csk]
   [clojure.string :as str])
  (:import
   (com.google.api.services.displayvideo.v2 DisplayVideo DisplayVideo$Partners$Channels$List)
   (com.google.api.services.displayvideo.v2.model Channel)))

(defn- fetch-channels-page
  [^DisplayVideo$Partners$Channels$List request ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (sdk-client-utils/execute-and-convert)))

(defn create
  [^DisplayVideo client ^long partner-id channel]
  (let [channel-conformed (spec/check channel ::displayvideo/channel)
        channel-request   (java-utils/clj->java Channel channel-conformed)]
    (-> client
        (.partners)
        (.channels)
        (.create partner-id channel-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo/channel))))

(defn get-by-id
  [^DisplayVideo client ^long partner-id ^long channel-id]
  (-> client
      (.partners)
      (.channels)
      (.get partner-id channel-id)
      (sdk-client-utils/execute-and-convert)
      (spec/check ::displayvideo/channel)))

(defn update-by-id
  [^DisplayVideo client
   ^Long partner-id
   ^Long channel-id
   channel
   update-mask]
  (let [mask-conformed    (spec/check update-mask ::displayvideo/update-mask)
        mask              (->> mask-conformed
                               (sequence (map csk/->camelCaseString))
                               (str/join \,))
        channel-request (java-utils/clj->java Channel channel)]
    (-> client
        (.partners)
        (.channels)
        (.patch partner-id channel-id channel-request)
        (.setUpdateMask mask)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo/channel))))

(defn get-list
  [^DisplayVideo client
   ^long partner-id
   {:keys [order-by filter-expr limit]}]
  (let [request (-> client
                    (.partners)
                    (.channels)
                    (.list partner-id))]
    (when order-by
      (.setOrderBy ^DisplayVideo$Partners$Channels$List request order-by))
    (when filter-expr
      (.setFilter ^DisplayVideo$Partners$Channels$List request filter-expr))
    (-> (sdk-client-utils/execute-and-convert-paginated
         (partial fetch-channels-page request)
         :channels
         limit)
        (spec/check ::displayvideo/channels))))