(ns com.adgoji.advertisers-channels.core
  (:require
   [com.adgoji.displayvideo-spec.interface :as displayvideo]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.sdk-client-utils.interface :as sdk-client-utils]
   [com.adgoji.spec.interface :as spec]
   [camel-snake-kebab.core :as csk]
   [clojure.string :as str])
  (:import
   (com.google.api.services.displayvideo.v2 DisplayVideo DisplayVideo$Advertisers$Channels$List)
   (com.google.api.services.displayvideo.v2.model Channel)))

(defn- fetch-channels-page
  [^DisplayVideo$Advertisers$Channels$List request ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (sdk-client-utils/execute-and-convert)))

(defn create
  [^DisplayVideo client ^long advertiser-id channel]
  (let [channel-conformed (spec/check channel ::displayvideo/channel)
        channel-request   (java-utils/clj->java Channel channel-conformed)]
    (-> client
        (.advertisers)
        (.channels)
        (.create advertiser-id channel-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo/channel))))

(defn get-by-id
  [^DisplayVideo client ^long advertiser-id ^long channel-id]
  (-> client
      (.advertisers)
      (.channels)
      (.get advertiser-id channel-id)
      (sdk-client-utils/execute-and-convert)
      (spec/check ::displayvideo/channel)))

(defn update-by-id
  [^DisplayVideo client
   ^Long advertiser-id
   ^Long channel-id
   channel
   update-mask]
  (let [mask-conformed    (spec/check update-mask ::displayvideo/update-mask)
        mask              (->> mask-conformed
                               (sequence (map csk/->camelCaseString))
                               (str/join \,))
        channel-request (java-utils/clj->java Channel channel)]
    (-> client
        (.advertisers)
        (.channels)
        (.patch advertiser-id channel-id channel-request)
        (.setUpdateMask mask)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo/channel))))

(defn get-list
  [^DisplayVideo client
   ^long advertiser-id
   {:keys [order-by filter-expr limit]}]
  (let [request (-> client
                    (.advertisers)
                    (.channels)
                    (.list advertiser-id))]
    (when order-by
      (.setOrderBy ^DisplayVideo$Advertisers$Channels$List request order-by))
    (when filter-expr
      (.setFilter ^DisplayVideo$Advertisers$Channels$List request filter-expr))
    (-> (sdk-client-utils/execute-and-convert-paginated
         (partial fetch-channels-page request)
         :channels
         limit)
        (spec/check ::displayvideo/channels))))