(ns com.adgoji.advertisers-channels-sites.core
  (:require
   [com.adgoji.displayvideo-spec.interface :as displayvideo]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.sdk-client-utils.interface :as sdk-client-utils]
   [com.adgoji.spec.interface :as spec])
  (:import
   (com.google.api.services.displayvideo.v2 DisplayVideo DisplayVideo$Advertisers$Channels$Sites$List)
   (com.google.api.services.displayvideo.v2.model BulkEditSitesRequest)
   (com.google.api.services.displayvideo.v2.model ReplaceSitesRequest)
   (com.google.api.services.displayvideo.v2.model Site)))

(defn- fetch-sites-page
  [^DisplayVideo$Advertisers$Channels$Sites$List request ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (sdk-client-utils/execute-and-convert)))

(defn bulk-edit
  [^DisplayVideo client advertiser-id channel-id bulk-edit-request]
  (let [bulk-edit-sites-conformed
        (spec/check bulk-edit-request ::displayvideo/bulk-edit-sites-request)

        bulk-edit-sites-request
        (java-utils/clj->java BulkEditSitesRequest bulk-edit-sites-conformed)]
    (-> client
        (.advertisers)
        (.channels)
        (.sites)
        (.bulkEdit (long advertiser-id)
                   (long channel-id)
                   bulk-edit-sites-request)
        (sdk-client-utils/execute-and-convert)
        :sites
        (spec/check ::displayvideo/sites))))

(defn create
  [^DisplayVideo client advertiser-id channel-id site]
  (let [site-conformed (spec/check site ::displayvideo/site)
        site-request   (java-utils/clj->java Site site-conformed)]
    (-> client
        (.advertisers)
        (.channels)
        (.sites)
        (.create (long advertiser-id) (long channel-id) site-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo/site))))

(defn delete
  [^DisplayVideo client advertiser-id channel-id url-or-app-id]
  (-> client
      (.advertisers)
      (.channels)
      (.sites)
      (.delete (long advertiser-id) (long channel-id) url-or-app-id)
      (sdk-client-utils/execute-and-convert)))

(defn replace-sites
  [^DisplayVideo client advertiser-id channel-id replace-sites]
  (let [replace-sites-conformed
        (spec/check replace-sites ::displayvideo/replace-sites-request)

        replace-sites-request
        (java-utils/clj->java ReplaceSitesRequest replace-sites-conformed)]
    (-> client
        (.advertisers)
        (.channels)
        (.sites)
        (.replace (long advertiser-id) (long channel-id) replace-sites-request)
        (sdk-client-utils/execute-and-convert)
        :sites
        (spec/check ::displayvideo/sites))))

(defn get-list
  [^DisplayVideo client
   advertiser-id
   channel-id
   {:keys [order-by filter-expr limit]}]
  (let [request (-> client
                    (.advertisers)
                    (.channels)
                    (.sites)
                    (.list (long advertiser-id) (long channel-id)))]
    (when order-by
      (.setOrderBy ^DisplayVideo$Advertisers$Channels$Sites$List request order-by))
    (when filter-expr
      (.setFilter ^DisplayVideo$Advertisers$Channels$Sites$List request filter-expr))
    (-> (sdk-client-utils/execute-and-convert-paginated
         (partial fetch-sites-page request)
         :sites
         limit)
        (spec/check ::displayvideo/sites))))
