(ns com.adgoji.api.displayvideo.advertisers
  (:require
   [com.adgoji.api.utils.api :as utils.api]
   [com.adgoji.api.utils.java :as utils.java]
   [clojure.string :as str]
   [camel-snake-kebab.core :as csk])
  (:import
   (com.google.api.services.displayvideo.v2 DisplayVideo DisplayVideo$Advertisers$List)
   (com.google.api.services.displayvideo.v2.model Advertiser)))

(set! *warn-on-reflection* true)

(defn- fetch-advertisers-page
  [^DisplayVideo$Advertisers$List request ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (utils.api/execute-and-convert)))

(defn get-list
  "Returns a list of available advertisers by `partner-id`."
  [^DisplayVideo client partner-id {:keys [order-by filter-expr limit]}]
  (let [request (-> client
                    (.advertisers)
                    (.list)
                    (.setPartnerId partner-id))]
    (when order-by
      (.setOrderBy ^DisplayVideo$Advertisers$List request order-by))
    (when filter-expr
      (.setFilter ^DisplayVideo$Advertisers$List request filter-expr))
    (utils.api/execute-and-convert-paginated (partial fetch-advertisers-page request)
                                             :advertisers
                                             limit)))

(defn get-by-id
  "Returns a single advertiser by `advertiser-id`."
  [^DisplayVideo client advertiser-id]
  (-> client
      (.advertisers)
      (.get advertiser-id)
      (utils.api/execute-and-convert)))

(defn create
  "Create a new advertiser."
  [^DisplayVideo client advertiser]
  (let [advertiser-request (utils.java/clj->java Advertiser advertiser)]
    (-> client
        (.advertisers)
        (.create advertiser-request)
        (utils.api/execute-and-convert))))

(defn delete
  "Delete an advertives by `advertiser-id`."
  [^DisplayVideo client advertiser-id]
  (-> client
      (.advertisers)
      (.delete advertiser-id)
      (utils.api/execute-and-convert)))

(defn update-by-id
  "Update an existing advertiser by `advertiser-id`.

  Request body is the same as for [[create]] function. Additionally
  `update-mask` is a mandatory parameter, it should be a sequence of
  fully qualified names of fields (as keywords).

  Example:
  ```
  [:display-name general-config.domain-url]
  ```."
  [^DisplayVideo client advertiser-id advertiser update-mask]
  (let [mask               (->> update-mask
                                (sequence (map csk/->camelCaseString))
                                (str/join \,))
        advertiser-request (utils.java/clj->java Advertiser advertiser)]
    (-> client
        (.advertisers)
        (.patch advertiser-id advertiser-request)
        (.setUpdateMask mask)
        (utils.api/execute-and-convert))))
