(ns com.adgoji.java-utils.core
  (:require
   [camel-snake-kebab.core :as csk]
   [clojure.java.data :as j])
  (:import
   (java.lang.reflect Method ParameterizedType)))

(defmulti clj->java
  "Adapted version of [[clojure.java.data/to-java]].

  Google models are not proper java beans and `to-java` function
  cannot find setter methods successfully."
  (fn [destination-type value] [destination-type (class value)]))

(defn- find-setter-method
  [^Class cls ^String method-name]
  (when-let [methods
             (->> (.getMethods cls)
                  (sequence (comp (filter (fn [^Method method]
                                            (= (.getName method) method-name)))
                                  (filter (fn [^Method method]
                                            (= 1 (count (.getParameterTypes method)))))))
                  (seq))]
    (if (not= 1 (count methods))
      (throw (ex-info (format "More than one %s method found" method-name) {}))
      (let [method           ^Method (first methods)
            parameters-count (count (.getParameterTypes method))]
        (if (not= 1 parameters-count)
          (throw (ex-info (format "Setter %s should take 1 argument, but it takes %d"
                                  method-name
                                  parameters-count)
                          {}))
          method)))))

(defn- get-setter-type
  ^Class
  [^Method method]
  (get (.getParameterTypes method) 0))

(defn- get-list-setter-type
  ^Class
  [^Method method]
  (let [list-type (-> method
                      (.getGenericParameterTypes)
                      (get 0))]
    (if (instance? ParameterizedType list-type)
      (-> ^ParameterizedType list-type
          (.getActualTypeArguments)
          (get 0))
      Object)))

(defn- maybe-convert-string
  "Return an appropriate Java class, created from `value` using
  `fromString` static method.

  If `fromString` method is not found, return `value` as is.  This is
  useful for 'broken' enums in SOAP objects, which are not really
  enums, but instances of classes."
  [^Class setter-type value]
  (if (and (string? value)
           (not (= java.lang.String setter-type)))
    (if-let [methods
             (->> (.getMethods setter-type)
                  (sequence (comp (filter (fn [^Method method]
                                            (= (.getName method) "fromString")))
                                  (filter (fn [^Method method]
                                            (= 1 (count (.getParameterTypes method)))))))
                  (seq))]
      (if (not= 1 (count methods))
        (throw (ex-info "More than one fromString method found" {}))
        (let [method ^Method (first methods)]
          (.invoke method setter-type (into-array [value]))))
      value)
    value))

(defn- convert-value
  [^Method setter-method ^Class setter-type value]
  (cond
    (= java.util.List setter-type)
    (let [list-type (get-list-setter-type setter-method)]
      (into [] (map (fn [item] (clj->java list-type item))) value))
    (.isArray setter-type)
    (let [list-type (.getComponentType setter-type)
          converted (into []
                          (map (fn [item]
                                 (->> item
                                      (maybe-convert-string list-type)
                                      (clj->java list-type))))
                          value)]
      (into-array list-type converted))
    :else
    (clj->java setter-type (maybe-convert-string setter-type value))))

(defn- set-prop
  [^Object instance [k v]]
  (let [clazz       (.getClass instance)
        setter-name (str "set" (csk/->PascalCaseString k))]
    (if-let [setter-method ^Method (find-setter-method clazz setter-name)]
      (let [setter-type     (get-setter-type setter-method)
            converted-value (convert-value setter-method setter-type v)]
        (if (or (nil? converted-value)
                (isa? (type converted-value) setter-type))
          ;; Some Java setters return the instance, some not.  We need
          ;; to make sure we always return the instance back.
          (or (.invoke setter-method instance (into-array [converted-value]))
              instance)
          (throw (ex-info "Incorrect value type"
                          {:setter-name setter-name
                           :class       clazz
                           :value       converted-value
                           :value-type  (type converted-value)
                           :setter-type setter-type}))))
      (throw (ex-info "Setter not found"
                      {:setter-name setter-name
                       :class       clazz})))))

(defmethod clj->java :default [^Class cls value]
  (j/to-java cls value))

(defmethod clj->java [java.lang.Object clojure.lang.APersistentMap]
  [^Class clazz props]
  (let [instance (.newInstance clazz)]
    (reduce set-prop instance props)))
