(ns com.adgoji.cm-api.reports.files
  (:require
   [com.adgoji.cm-reports-files.interface :as cm-reports-files]
   [clojure.spec.alpha :as s]
   [com.adgoji.cm-spec.interface :as cm-spec]))

(defn get-list
  "Returns list of files for a report.

  The result can be altered using various `opts`, which are described
  in the official
  documentation: [[https://developers.google.com/doubleclick-advertisers/rest/v4/reports.files/list#query-parameters]].

  Options should be provided as a map with kebab-case keys."
  [client profile-id report-id opts]
  (cm-reports-files/get-list client profile-id report-id opts))

(s/fdef get-list
  :args (s/cat :client ::cm-spec/client
               :profile-id ::cm-spec/profile-id
               :report-id ::cm-spec/report-id
               :opts (s/keys :opt-un [::cm-spec/max-results
                                      ::cm-spec/sort-field
                                      ::cm-spec/sort-order]))
  :ret ::cm-spec/files)

(defn get-by-id
  "Returns a report file by provided `report-id` and `file-id`.

  By default this function returns a report file status. With
  additional `media?` argument it's possible to fetch a file as an
  input stream.

  NOTE: Closing input stream is a caller responsibility. Usage
  example:

  ```clojure
  (with-open [is (get-by-id client profile-id report-id file-id true)]
    (read-from-input-stream is))
  ```"
  ([client profile-id report-id file-id]
   (get-by-id client profile-id report-id file-id false))
  ([client profile-id report-id file-id media?]
   (cm-reports-files/get-by-id client
                               profile-id
                               report-id
                               file-id
                               media?)))

(s/fdef get-by-id
  :args (s/or :no-media (s/cat :client ::cm-spec/client
                               :profile-id ::cm-spec/profile-id
                               :report-id ::cm-spec/report-id
                               :file-id ::cm-spec/file-id)
              :media (s/cat :client ::cm-spec/client
                            :profile-id ::cm-spec/profile-id
                            :report-id ::cm-spec/report-id
                            :file-id ::cm-spec/file-id
                            :media? boolean?))
  :ret ::cm-spec/file)
