(ns com.adgoji.cm-reports.core
  (:require
   [clojure.string :as str]
   [com.adgoji.cm-spec.interface :as cm-spec]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.sdk-client-utils.interface :as sdk-client-utils]
   [com.adgoji.spec.interface :as spec])
  (:import
   (com.google.api.services.dfareporting Dfareporting Dfareporting$Reports$List)
   (com.google.api.services.dfareporting.model Report)))

(defn- fetch-reports-page
  [^Dfareporting$Reports$List request ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (sdk-client-utils/execute-and-convert)
      ;; For some reason this API returns `:next-page-token` as an
      ;; empty string.
      (update :next-page-token #(if (str/blank? %) nil %))))

(defn get-list
  [^Dfareporting client
   profile-id
   {:keys [max-results
           scope
           sort-field
           sort-order]}]
  (let [profile-id-conformed (spec/check profile-id ::cm-spec/profile-id)
        request              (-> client
                                 (.reports)
                                 (.list profile-id-conformed))]
    (when max-results
      (.setMaxResults ^Dfareporting$Reports$List request
                      (spec/check max-results ::cm-spec/max-results)))
    (when scope
      (.setScope ^Dfareporting$Reports$List request
                 (spec/check scope ::cm-spec/scope)))
    (when sort-field
      (.setSortField ^Dfareporting$Reports$List request
                     (spec/check sort-field ::cm-spec/sort-field)))
    (when sort-order
      (.setSortOrder ^Dfareporting$Reports$List request
                     (spec/check sort-order ::cm-spec/sort-order)))
    (-> (sdk-client-utils/execute-and-convert-paginated
         (partial fetch-reports-page request)
         :items
         max-results)
        (spec/check ::cm-spec/reports))))

(defn get-by-id
  [^Dfareporting client profile-id report-id]
  (-> client
      (.reports)
      (.get (spec/check profile-id ::cm-spec/profile-id)
            (spec/check report-id ::cm-spec/report-id))
      (sdk-client-utils/execute-and-convert)
      (spec/check ::cm-spec/report)))

(defn insert
  [^Dfareporting client profile-id report]
  (let [report-conformed (spec/check report ::cm-spec/report)
        report-request   (java-utils/clj->java Report report-conformed)]
    (-> client
        (.reports)
        (.insert (spec/check profile-id ::cm-spec/profile-id)
                 report-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::cm-spec/report))))

(defn patch
  [^Dfareporting client profile-id report-id content]
  (let [content-conformed (spec/check content ::cm-spec/report-update)
        content-request   (java-utils/clj->java Report content-conformed)]
    (-> client
        (.reports)
        (.patch (spec/check profile-id ::cm-spec/profile-id)
                (spec/check report-id ::cm-spec/report-id)
                content-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::cm-spec/report))))

(defn put
  [^Dfareporting client profile-id report-id report]
  (let [content-conformed (spec/check report ::cm-spec/report)
        content-request   (java-utils/clj->java Report content-conformed)]
    (-> client
        (.reports)
        (.update (spec/check profile-id ::cm-spec/profile-id)
                 (spec/check report-id ::cm-spec/report-id)
                 content-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::cm-spec/report))))

(defn delete
  [^Dfareporting client profile-id report-id]
  (-> client
      (.reports)
      (.delete (spec/check profile-id ::cm-spec/profile-id)
               (spec/check report-id ::cm-spec/report-id))
      (sdk-client-utils/execute-and-convert)))

(defn run
  [^Dfareporting client
   profile-id
   report-id
   {:keys [synchronous]
    :or   {synchronous false}}]
  (-> client
      (.reports)
      (.run (spec/check profile-id ::cm-spec/profile-id)
            (spec/check report-id ::cm-spec/report-id))
      (.setSynchronous (spec/check synchronous ::cm-spec/synchronous))
      (sdk-client-utils/execute-and-convert)
      (spec/check ::cm-spec/file)))
