(ns com.adgoji.google-credential.core
  (:require
   [clojure.java.io :as io])
  (:import
   (com.google.api.client.auth.oauth2 Credential)
   (com.google.api.client.extensions.java6.auth.oauth2 AuthorizationCodeInstalledApp)
   (com.google.api.client.extensions.jetty.auth.oauth2 LocalServerReceiver)
   (com.google.api.client.googleapis.auth.oauth2 GoogleAuthorizationCodeFlow$Builder GoogleClientSecrets GoogleCredential)
   (com.google.api.client.googleapis.util Utils)
   (java.util Base64)))

(defn- decode ^String [^String s]
  (-> (Base64/getDecoder)
      (.decode s)
      (String.)))

(defmulti new :mode :default ::default)

(defmethod new :service-base64
  ^Credential
  [{:keys [credential scopes]}]
  (with-open [credential-bytes (-> credential
                                   (decode)
                                   (.getBytes)
                                   (io/input-stream))]
    (-> credential-bytes
        (GoogleCredential/fromStream)
        (.createScoped scopes))))

(defmethod new :service-file
  ^Credential
  [{:keys [credential scopes]}]
  (with-open [credential-file (io/input-stream (io/file credential))]
    (-> credential-file
        (GoogleCredential/fromStream)
        (.createScoped scopes))))

(defmethod new :installed
  [{:keys [credential scopes]}]
  ^Credential
  (with-open [credentials-file (io/reader (io/file credential))]
    (let [secrets (GoogleClientSecrets/load (Utils/getDefaultJsonFactory) credentials-file)
          flow    (-> (GoogleAuthorizationCodeFlow$Builder.
                       (Utils/getDefaultTransport)
                       (Utils/getDefaultJsonFactory)
                       secrets
                       scopes)
                      (.build))]
      (-> (AuthorizationCodeInstalledApp. flow (LocalServerReceiver.))
          (.authorize "user")))))

(defmethod new ::default
  [{:keys [mode]}]
  (throw (ex-info "Unknown mode" {:mode mode})))
