(ns com.adgoji.cm-api.reports
  (:require
   [com.adgoji.cm-reports.interface :as cm-reports]
   [clojure.spec.alpha :as s]
   [com.adgoji.cm-spec.interface :as cm-spec]))

(defn get-list
  "Returns list of reports.

  The result can be altered using various `opts`, which are described
  in the official
  documentation: [[https://developers.google.com/doubleclick-advertisers/rest/v4/reports/list#query-parameters]].

  Options should be provided as a map with kebab-case keys."
  [client profile-id opts]
  (cm-reports/get-list client profile-id opts))

(s/fdef get-list
  :args (s/cat :client ::cm-spec/client
               :profile-id ::cm-spec/profile-id
               :opts (s/keys :opt-un [::cm-spec/max-results
                                      ::cm-spec/scope
                                      ::cm-spec/sort-field
                                      ::cm-spec/sort-order]))
  :ret ::cm-spec/reports)

(defn insert
  "Creates a report."
  [client profile-id report]
  (cm-reports/insert client profile-id report))

(s/fdef insert
  :args (s/cat :client ::cm-spec/client
               :profile-id ::cm-spec/profile-id
               :report ::cm-spec/report)
  :ret ::cm-spec/report)

(defn get-by-id
  "Returns a report by provided `profile-id` and `report-id`."
  [client profile-id report-id]
  (cm-reports/get-by-id client profile-id report-id))

(s/fdef get-by-id
  :args (s/cat :client ::cm-spec/client
               :profile-id ::cm-spec/profile-id
               :report-id ::cm-spec/report-id)
  :ret ::cm-spec/report)

(defn patch
  "Updates an existing report.

  This method supports patch semantics."
  [client profile-id report-id content]
  (cm-reports/patch client profile-id report-id content))

(s/fdef patch
  :args (s/cat :client ::cm-spec/client
               :profile-id ::cm-spec/profile-id
               :report-id ::cm-spec/report-id
               :content ::cm-spec/report-update)
  :ret ::cm-spec/report)

(defn put
  "Updates an existing report."
  [client profile-id report-id content]
  (cm-reports/put client profile-id report-id content))

(s/fdef put
  :args (s/cat :client ::cm-spec/client
               :profile-id ::cm-spec/profile-id
               :report-id ::cm-spec/report-id
               :content ::cm-spec/report)
  :ret ::cm-spec/report)

(defn delete
  "Deletes a report by provided `profile-id` and `report-id`."
  [client profile-id report-id]
  (cm-reports/delete client profile-id report-id))

(s/fdef delete
  :args (s/cat :client ::cm-spec/client
               :profile-id ::cm-spec/profile-id
               :report-id ::cm-spec/report-id)
  :ret nil?)

(defn run
  "Runs a report.

  By default report is run asynchronously, this can be altered by
  providing option `:synchronous true` as a last argument."
  [client profile-id report-id opts]
  (cm-reports/run client profile-id report-id opts))

(s/fdef run
  :args (s/cat :client ::cm-spec/client
               :profile-id ::cm-spec/profile-id
               :report-id ::cm-spec/report-id
               :opts (s/keys :opt-un [::cm-spec/synchronous]))
  :ret ::cm-spec/file)
