(ns com.adgoji.cm-advertisers.core
  (:require
   [com.adgoji.cm-spec.interface :as cm-spec]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.sdk-client-utils.interface :as sdk-client-utils]
   [com.adgoji.spec.interface :as spec])
  (:import
   (com.google.api.services.dfareporting Dfareporting Dfareporting$Advertisers$List)
   (com.google.api.services.dfareporting.model Advertiser)))

(defn- fetch-advertisers-page
  [^Dfareporting$Advertisers$List request ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (sdk-client-utils/execute-and-convert)))

(defn get-list
  "Returns a list of available advertisers by `profile-id`."
  [^Dfareporting client
   profile-id
   {:keys [max-results
           subaccount-id
           status
           advertiser-group-ids
           floodlight-configuration-ids
           include-advertisers-without-groups-only
           only-parent
           ids
           search-string
           sort-field
           sort-order]}]
  (let [profile-id-conformed (spec/check profile-id ::cm-spec/profile-id)
        request              (-> client
                                 (.advertisers)
                                 (.list profile-id-conformed))]
    (when max-results
      (.setMaxResults ^Dfareporting$Advertisers$List request
                      (spec/check max-results ::cm-spec/max-results)))
    (when subaccount-id
      (.setSubaccountId ^Dfareporting$Advertisers$List request
                        (spec/check subaccount-id
                                    ::cm-spec/subaccount-id)))
    (when status
      (.setStatus ^Dfareporting$Advertisers$List request
                  (spec/check status
                              ::cm-spec/advertiser-status)))
    (when (seq advertiser-group-ids)
      (.setAdvertiserGroupIds ^Dfareporting$Advertisers$List request
                              (spec/check advertiser-group-ids
                                          ::cm-spec/advertiser-group-ids)))
    (when (seq floodlight-configuration-ids)
      (.setFloodlightConfigurationIds ^Dfareporting$Advertisers$List request
                                      (spec/check floodlight-configuration-ids
                                                  ::cm-spec/floodlight-configuration-ids)))
    (when (some? include-advertisers-without-groups-only)
      (.setIncludeAdvertisersWithoutGroupsOnly ^Dfareporting$Advertisers$List request
                                               (spec/check include-advertisers-without-groups-only
                                                           ::cm-spec/include-advertisers-without-groups-only)))
    (when (some? only-parent)
      (.setOnlyParent ^Dfareporting$Advertisers$List request
                      (spec/check only-parent ::cm-spec/only-parent)))
    (when (seq ids)
      (.setIds ^Dfareporting$Advertisers$List request
               (spec/check ids ::cm-spec/ids)))

    (when search-string
      (.setSearchString ^Dfareporting$Advertisers$List request
                        (spec/check search-string ::cm-spec/search-string)))
    (when sort-field
      (.setSortField ^Dfareporting$Advertisers$List request
                     (spec/check sort-field ::cm-spec/sort-field)))
    (when sort-order
      (.setSortOrder ^Dfareporting$Advertisers$List request
                     (spec/check sort-order ::cm-spec/sort-order)))
    (-> (sdk-client-utils/execute-and-convert-paginated
         (partial fetch-advertisers-page request)
         :advertisers
         max-results)
        (spec/check ::cm-spec/advertisers))))

(defn get-by-id
  [^Dfareporting client profile-id advertiser-id]
  (-> client
      (.advertisers)
      (.get (spec/check profile-id ::cm-spec/profile-id)
            (spec/check advertiser-id ::cm-spec/advertiser-id))
      (sdk-client-utils/execute-and-convert)
      (spec/check ::cm-spec/advertiser)))

(defn insert
  [^Dfareporting client profile-id advertiser]
  (let [advertiser-conformed (spec/check advertiser ::cm-spec/advertiser)
        advertiser-request   (java-utils/clj->java Advertiser
                                                   advertiser-conformed)]
    (-> client
        (.advertisers)
        (.insert (spec/check profile-id ::cm-spec/profile-id)
                 advertiser-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::cm-spec/advertiser))))

(defn patch
  [^Dfareporting client profile-id advertiser-id content]
  (let [content-conformed (spec/check content ::cm-spec/advertiser-update)
        content-request   (java-utils/clj->java Advertiser content-conformed)]
    (-> client
        (.advertisers)
        (.patch (spec/check profile-id ::cm-spec/profile-id)
                (spec/check advertiser-id ::cm-spec/profile-id)
                content-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::cm-spec/advertiser))))

(defn put
  [^Dfareporting client profile-id advertiser]
  (let [content-conformed (spec/check advertiser ::cm-spec/advertiser)
        content-request   (java-utils/clj->java Advertiser content-conformed)]
    (-> client
        (.advertisers)
        (.update (spec/check profile-id ::cm-spec/profile-id)
                 content-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::cm-spec/advertiser))))
