(ns com.adgoji.sdk-client-utils.core
  (:require
   [camel-snake-kebab.core :as csk]
   [camel-snake-kebab.extras :as cske]
   [clojure.core.memoize :as m]
   [clojure.java.data :as j]
   [clojure.string :as str]
   [clojure.walk :as walk])
  (:import
   (com.google.api.client.googleapis.json GoogleJsonResponseException)
   (com.google.api.client.googleapis.services AbstractGoogleClientRequest)))

(def ^:private memoized->kebab-case-keyword
  (m/fifo (fn [input]
            (csk/->kebab-case-keyword input :separator #"(?<![A-Z])(?=[A-Z])"))
          {}
          :fifo/threshold 512))

(defn- transform-keys
  [response]
  (cske/transform-keys memoized->kebab-case-keyword response))

(defn- ensure-vecs
  [obj]
  (walk/postwalk (fn [node]
                   (if (seq? node)
                     (into [] node)
                     node))
                 obj))

(defn convert
  [obj]
  (-> obj
      (j/from-java-deep {})
      (ensure-vecs)
      (transform-keys)))

(defn execute
  [^AbstractGoogleClientRequest request]
  (.execute request))

(defn execute-and-convert
  [^AbstractGoogleClientRequest request]
  (try
    (-> request
        (execute)
        (convert))
    (catch GoogleJsonResponseException ex
      (throw (ex-info (.getStatusMessage ex)
                      (convert (.getDetails ex))
                      ex)))))

(defn execute-and-convert-paginated
  [fetch-page-fn vkey limit]
  (let [xform (if limit
                (comp cat (take limit))
                cat)]
    (->> (iteration fetch-page-fn
                    :kf :next-page-token
                    :vf vkey
                    :initk nil)
         (into [] xform))))

(defn normalize-mask
  [mask]
  (->> mask
       (sequence (map csk/->camelCaseString))
       (str/join \,)))
