(ns com.adgoji.ad-manager-line-items.core
  (:require
   [com.adgoji.ad-manager-client.interface :as ad-manager-client]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.soap-client-utils.interface :as soap-client-utils])
  (:import
   (com.google.api.ads.admanager.axis.utils.v202502 StatementBuilder)
   (com.google.api.ads.admanager.axis.v202502 ActivateLineItems ArchiveLineItems DeleteLineItems LineItem LineItemPage LineItemServiceInterface PauseLineItems ReleaseLineItems ReserveAndOverbookLineItems ReserveLineItems ResumeAndOverbookLineItems ResumeLineItems UnarchiveLineItems)))

(defn create
  [client line-items]
  (let [service (ad-manager-client/service client :line-item)
        body    (->> line-items
                     (sequence (map (partial java-utils/clj->java LineItem)))
                     (into-array LineItem))]
    (soap-client-utils/execute-and-convert LineItemServiceInterface/.createLineItems
                                           service
                                           body)))

(defn patch
  [client line-items]
  (let [service (ad-manager-client/service client :line-item)
        body    (->> line-items
                     (sequence (map (partial java-utils/clj->java LineItem)))
                     (into-array LineItem))]
    (soap-client-utils/execute-and-convert LineItemServiceInterface/.updateLineItems
                                           service
                                           body)))

(defn- fetch-line-items-page
  [^LineItemServiceInterface service
   ^StatementBuilder statement-builder
   offset]
  (.increaseOffsetBy statement-builder (int offset))
  (soap-client-utils/execute-and-convert
   (comp LineItemPage/.getResults
         LineItemServiceInterface/.getLineItemsByStatement)
   service
   (.toStatement statement-builder)))

(defn all
  [client statement {:keys [limit]}]
  (let [service   (ad-manager-client/service client :line-item)
        page-size StatementBuilder/SUGGESTED_PAGE_LIMIT
        statement (-> (soap-client-utils/statement-builder statement)
                      (.limit page-size))]
    (soap-client-utils/execute-and-convert-paginated
     (partial fetch-line-items-page service statement)
     page-size
     {:limit limit})))

(def ^:private line-item-action
  {:activate-line-items             (ActivateLineItems/new)
   :archive-line-items              (ArchiveLineItems/new)
   :delete-line-items               (DeleteLineItems/new)
   :pause-line-items                (PauseLineItems/new)
   :release-line-items              (ReleaseLineItems/new)
   :reserve-line-items              (ReserveLineItems/new)
   :reserve-and-overbook-line-items (ReserveAndOverbookLineItems/new)
   :resume-line-items               (ResumeLineItems/new)
   :resume-and-overbook-line-items  (ResumeAndOverbookLineItems/new)
   :unarchive-line-items            (UnarchiveLineItems/new)})

(defn perform-action
  [client line-item-id action]
  (let [service   (ad-manager-client/service client :line-item)
        statement (-> (soap-client-utils/statement-builder
                       {:where [[:= :id line-item-id]]})
                      (.toStatement))]
    (soap-client-utils/execute-and-convert
     LineItemServiceInterface/.performLineItemAction
     service
     (line-item-action action)
     statement)))
