(ns com.adgoji.ad-manager-users.core
  (:require
   [com.adgoji.ad-manager-client.interface :as ad-manager-client]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.soap-client-utils.interface :as soap-client-utils])
  (:import
   (com.google.api.ads.admanager.axis.utils.v202502 StatementBuilder)
   (com.google.api.ads.admanager.axis.v202502 ActivateUsers DeactivateUsers User UserPage UserServiceInterface)))

(defn all-roles
  [client]
  (let [service (ad-manager-client/service client :user)]
    (soap-client-utils/execute-and-convert UserServiceInterface/.getAllRoles
                                           service)))

(defn create
  [client users]
  (let [service (ad-manager-client/service client :user)
        body    (->> users
                     (sequence (map (partial java-utils/clj->java User)))
                     (into-array User))]
    (soap-client-utils/execute-and-convert UserServiceInterface/.createUsers
                                           service
                                           body)))

(defn patch
  [client users]
  (let [service (ad-manager-client/service client :user)
        body    (->> users
                     (sequence (map (partial java-utils/clj->java User)))
                     (into-array User))]
    (soap-client-utils/execute-and-convert UserServiceInterface/.updateUsers
                                           service
                                           body)))

(defn current
  [client]
  (let [service (ad-manager-client/service client :user)]
    (soap-client-utils/execute-and-convert UserServiceInterface/.getCurrentUser
                                           service)))

(defn- fetch-users-page
  [^UserServiceInterface service ^StatementBuilder statement-builder offset]
  (.increaseOffsetBy statement-builder (int offset))
  (soap-client-utils/execute-and-convert
   (comp UserPage/.getResults
         UserServiceInterface/.getUsersByStatement)
   service
   (.toStatement statement-builder)))

(defn all
  [client statement {:keys [limit]}]
  (let [service   (ad-manager-client/service client :user)
        page-size StatementBuilder/SUGGESTED_PAGE_LIMIT
        statement (-> (soap-client-utils/statement-builder statement)
                      (.limit page-size))]
    (soap-client-utils/execute-and-convert-paginated
     (partial fetch-users-page service statement)
     page-size
     {:limit limit})))

(def ^:private user-action
  {:activate-users   (ActivateUsers/new)
   :deactivate-users (DeactivateUsers/new)})

(defn perform-action
  [client user-id action]
  (let [service   (ad-manager-client/service client :user)
        statement (-> (soap-client-utils/statement-builder
                       {:where [[:= :id user-id]]})
                      (.toStatement))]
    (soap-client-utils/execute-and-convert
     UserServiceInterface/.performUserAction
     service
     (user-action action)
     statement)))
