(ns com.adgoji.ad-manager-ad-units.core
  (:require
   [com.adgoji.ad-manager-client.interface :as ad-manager-client]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.soap-client-utils.interface :as soap-client-utils])
  (:import
   (com.google.api.ads.admanager.axis.utils.v202502 StatementBuilder)
   (com.google.api.ads.admanager.axis.v202502 ActivateAdUnits AdUnit AdUnitPage ArchiveAdUnits DeactivateAdUnits InventoryServiceInterface Statement)))

(defn create
  [client ad-units]
  (let [service (ad-manager-client/service client :inventory)
        body    (->> ad-units
                     (sequence (map (partial java-utils/clj->java AdUnit)))
                     (into-array AdUnit))]
    (soap-client-utils/execute-and-convert InventoryServiceInterface/.createAdUnits
                                           service
                                           body)))

(defn patch
  [client ad-units]
  (let [service (ad-manager-client/service client :inventory)
        body    (->> ad-units
                     (sequence (map (partial java-utils/clj->java AdUnit)))
                     (into-array AdUnit))]
    (soap-client-utils/execute-and-convert InventoryServiceInterface/.updateAdUnits
                                           service
                                           body)))

(defn all-sizes
  [client]
  (let [service (ad-manager-client/service client :inventory)]
    (soap-client-utils/execute-and-convert
     InventoryServiceInterface/.getAdUnitSizesByStatement
     service
     (Statement/new))))

(defn- fetch-ad-units-page
  [^InventoryServiceInterface service ^StatementBuilder statement-builder offset]
  (.increaseOffsetBy statement-builder (int offset))
  (soap-client-utils/execute-and-convert (comp AdUnitPage/.getResults
                                               InventoryServiceInterface/.getAdUnitsByStatement)
                                         service
                                         (.toStatement statement-builder)))

(defn all
  [client statement {:keys [limit]}]
  (let [service   (ad-manager-client/service client :inventory)
        page-size StatementBuilder/SUGGESTED_PAGE_LIMIT
        statement (-> (soap-client-utils/statement-builder statement)
                      (.limit page-size))]
    (soap-client-utils/execute-and-convert-paginated
     (partial fetch-ad-units-page service statement)
     page-size
     {:limit limit})))

(def ^:private ad-unit-action
  {:activate-ad-units   (ActivateAdUnits/new)
   :archive-ad-units    (ArchiveAdUnits/new)
   :deactivate-ad-units (DeactivateAdUnits/new)})

(defn perform-action
  [client ad-unit-id action]
  (let [service   (ad-manager-client/service client :inventory)
        statement (-> (soap-client-utils/statement-builder
                       {:where [[:= :id ad-unit-id]]})
                      (.toStatement))]
    (soap-client-utils/execute-and-convert
     InventoryServiceInterface/.performAdUnitAction
     service
     (ad-unit-action action)
     statement)))
