(ns com.adgoji.ad-manager-pql.core
  (:require
   [clojure.string :as str]
   [com.adgoji.ad-manager-client.interface :as ad-manager-client]
   [com.adgoji.soap-client-utils.interface :as soap-client-utils]
   [clojure.java.data :as j])
  (:import
   (com.google.api.ads.admanager.axis.utils.v202502 StatementBuilder)
   (com.google.api.ads.admanager.axis.v202502 PublisherQueryLanguageServiceInterface ResultSet Row)))

(defn- extract-values-from-row
  [^Row row]
  (into []
        (comp (map j/from-java) (map :value))
        (.getValues row)))

(defn- extract-values-from-rows
  [rows]
  (into [] (map extract-values-from-row) rows))

(defn- fetch-pql-page
  [^PublisherQueryLanguageServiceInterface service
   ^StatementBuilder statement-builder
   offset]
  (.increaseOffsetBy statement-builder (int offset))
  (->> (soap-client-utils/execute-and-convert
        (comp extract-values-from-rows
              ResultSet/.getRows
              PublisherQueryLanguageServiceInterface/.select)
        service
        (.toStatement statement-builder))
       (not-empty)))

(defn- validate-query
  [query]
  (when-let [error-msg
             (cond-> []
               (not (seq (:select query))) (conj "select")
               (not (:from query))         (conj "from")
               :always                     not-empty)]
    (throw (ex-info (str "Missing statements: "
                         (str/join ", " error-msg))
                    query))))

(defn- rows->maps
  [header rows]
  (into []
        (map (partial zipmap header))
        rows))

(defn select
  [client query {:keys [limit]}]
  (validate-query query)
  (let [service           (ad-manager-client/service client :pql)
        page-size         StatementBuilder/SUGGESTED_PAGE_LIMIT
        statement-builder (soap-client-utils/statement-builder query)
        statement-rows    (.limit statement-builder page-size)
        header            (:select query)
        rows              (soap-client-utils/execute-and-convert-paginated
                           (partial fetch-pql-page service statement-rows)
                           page-size
                           {:limit limit})]
    (rows->maps header rows)))
