(ns com.adgoji.ad-manager-creatives.core
  (:require
   [com.adgoji.ad-manager-client.interface :as ad-manager-client]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.soap-client-utils.interface :as soap-client-utils])
  (:import
   (com.google.api.ads.admanager.axis.utils.v202502 StatementBuilder)
   (com.google.api.ads.admanager.axis.v202502 ActivateCreatives AdExchangeCreative AdSenseCreative AspectRatioImageCreative AudioCreative AudioRedirectCreative Creative CreativePage CreativeServiceInterface CustomCreative DeactivateCreatives Html5Creative ImageCreative ImageOverlayCreative ImageRedirectCreative ImageRedirectOverlayCreative InternalRedirectCreative LegacyDfpCreative ProgrammaticCreative RichMediaStudioCreative SetTopBoxCreative TemplateCreative ThirdPartyCreative UnsupportedCreative VastRedirectCreative VideoCreative VideoRedirectCreative)))

(def ^:private creative-types
  {:ad-exchange-creative            AdExchangeCreative
   :ad-sense-creative               AdSenseCreative
   :rich-media-studio-creative      RichMediaStudioCreative
   :aspect-ratio-image-creative     AspectRatioImageCreative
   :audio-creative                  AudioCreative
   :audio-redirect-creative         AudioRedirectCreative
   :image-creative                  ImageCreative
   :image-overlay-creative          ImageOverlayCreative
   :image-redirect-creative         ImageRedirectCreative
   :image-redirect-overlay-creative ImageRedirectOverlayCreative
   :set-top-box-creative            SetTopBoxCreative
   :video-creative                  VideoCreative
   :video-redirect-creative         VideoRedirectCreative
   :custom-creative                 CustomCreative
   :html5-creative                  Html5Creative
   :internal-redirect-creative      InternalRedirectCreative
   :legacy-dfp-creative             LegacyDfpCreative
   :programmatic-creative           ProgrammaticCreative
   :template-creative               TemplateCreative
   :third-party-creative            ThirdPartyCreative
   :unsupported-creative            UnsupportedCreative
   :vast-redirect-creative          VastRedirectCreative})

(defn- creative-instance
  [{creative-type :type :as creative}]
  (if-let [cls (creative-types creative-type)]
    (java-utils/clj->java cls (dissoc creative :type))
    (throw (ex-info "Unsupported or missing creative type"
                    {:creative-type creative-type}))))

(defn create
  [client creatives]
  (let [service (ad-manager-client/service client :creative)
        body    (->> creatives
                     (sequence (map creative-instance))
                     (into-array Creative))]
    (soap-client-utils/execute-and-convert CreativeServiceInterface/.createCreatives
                                           service
                                           body)))

(defn- fetch-creatives-page
  [^CreativeServiceInterface service
   ^StatementBuilder statement-builder
   offset]
  (.increaseOffsetBy statement-builder (int offset))
  (soap-client-utils/execute-and-convert
   (comp CreativePage/.getResults
         CreativeServiceInterface/.getCreativesByStatement)
   service
   (.toStatement statement-builder)))

(defn all
  [client statement {:keys [limit]}]
  (let [service   (ad-manager-client/service client :creative)
        page-size StatementBuilder/SUGGESTED_PAGE_LIMIT
        statement (-> (soap-client-utils/statement-builder statement)
                      (.limit page-size))]
    (soap-client-utils/execute-and-convert-paginated
     (partial fetch-creatives-page service statement)
     page-size
     {:limit limit})))

(def ^:private creative-action
  {:activate-creatives   (ActivateCreatives/new)
   :deactivate-creatives (DeactivateCreatives/new)})

(defn perform-action
  [client creative-id action]
  (let [service   (ad-manager-client/service client :creative)
        statement (-> (soap-client-utils/statement-builder
                       {:where [[:= :id creative-id]]})
                      (.toStatement))]
    (soap-client-utils/execute-and-convert
     CreativeServiceInterface/.performCreativeAction
     service
     (creative-action action)
     statement)))
