(ns com.adgoji.ad-manager-reports.core
  (:require
   [com.adgoji.ad-manager-client.interface :as ad-manager-client]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.soap-client-utils.interface :as soap-client-utils])
  (:import
   (com.google.api.ads.admanager.axis.utils.v202502 StatementBuilder)
   (com.google.api.ads.admanager.axis.v202502 ExportFormat ReportDownloadOptions ReportJob ReportServiceInterface SavedQueryPage Statement)))

(defn- job-statement
  "Convert `statement` map to [[Statement]] Java class instance."
  ^Statement
  [statement]
  (.toStatement (soap-client-utils/statement-builder statement)))

(defn run-job
  [client {{:keys [statement]} :report-query :as job}]
  (let [service            (ad-manager-client/service client :report)
        job-with-statement (cond-> job
                             statement (update-in [:report-query :statement] job-statement))
        body               (java-utils/clj->java ReportJob job-with-statement)]
    (soap-client-utils/execute-and-convert ReportServiceInterface/.runReportJob
                                           service
                                           body)))

(defn get-job-status
  [client job-id]
  (let [service (ad-manager-client/service client :report)]
    (soap-client-utils/execute-and-convert ReportServiceInterface/.getReportJobStatus
                                           service
                                           job-id)))

(defn get-download-url
  [client job-id export-format]
  (let [service (ad-manager-client/service client :report)]
    (soap-client-utils/execute-and-convert ReportServiceInterface/.getReportDownloadURL
                                           service
                                           job-id
                                           (ExportFormat/fromString export-format))))

(defn get-download-url-with-options
  [client job-id opts]
  (let [service       (ad-manager-client/service client :report)
        download-opts (java-utils/clj->java ReportDownloadOptions opts)]
    (soap-client-utils/execute-and-convert ReportServiceInterface/.getReportDownloadUrlWithOptions
                                           service
                                           job-id
                                           download-opts)))

(defn- fetch-queries-page
  [^ReportServiceInterface service ^StatementBuilder statement-builder offset]
  (.increaseOffsetBy statement-builder (int offset))
  (soap-client-utils/execute-and-convert (comp SavedQueryPage/.getResults
                                               ReportServiceInterface/.getSavedQueriesByStatement)
                                         service
                                         (.toStatement statement-builder)))

(defn get-saved-queries
  [client statement {:keys [limit]}]
  (let [service   (ad-manager-client/service client :report)
        page-size StatementBuilder/SUGGESTED_PAGE_LIMIT
        statement (-> (soap-client-utils/statement-builder statement)
                      (.limit page-size))]
    (soap-client-utils/execute-and-convert-paginated
     (partial fetch-queries-page service statement)
     page-size
     {:limit limit})))
