(ns com.adgoji.soap-client-utils.core
  (:require
   [camel-snake-kebab.core :as csk]
   [camel-snake-kebab.extras :as cske]
   [clojure.core.memoize :as m]
   [clojure.java.data :as j]
   [clojure.string :as str]
   [clojure.walk :as walk])
  (:import
   (com.google.api.ads.admanager.axis.utils.v202502 StatementBuilder)
   (com.google.api.ads.admanager.axis.v202502 ApiException)
   (java.time Instant ZonedDateTime ZoneId ZoneRegion)))

;;; Statement builder

(defn- bind-statement-value
  ^StatementBuilder
  [^StatementBuilder builder var-key var-value]
  (let [var-name ^String (csk/->camelCaseString var-key)]
    (.withBindVariableValue builder var-name var-value)))

(defn- bind-statement-values
  ^StatementBuilder
  [values]
  (reduce-kv bind-statement-value (StatementBuilder.) values))

(defn- select-expr
  [{:keys [select]}]
  (when (seq select)
    (str/join ", " (sequence (map csk/->PascalCaseString) select))))

(defn- from-expr
  [{:keys [from]}]
  (when from
    (csk/->Camel_Snake_Case_String from)))

(defn- format-where-statement
  [[op var-name var-value]]
  (if-not (and op var-name)
    (throw (ex-info "Invalid where expression"
                    {:op        op
                     :var-name  var-name
                     :var-value var-value}))
    (let [var-name-cc (csk/->camelCaseString var-name)]
      (format "%s %s :%s"
              var-name-cc
              (str/upper-case (name op))
              var-name-cc))))

(defn- where-expr
  [{:keys [where]}]
  (when (seq where)
    (->> where
         (sequence (comp (map format-where-statement)
                         (remove nil?)))
         (str/join " AND "))))

(defn- values-bindings
  [{:keys [where]}]
  (when (seq where)
    (into {}
          (map (fn [[_ var-name var-value]]
                 [var-name var-value]))
          where)))

(defn- order-by-expr
  [{:keys [order-by]}]
  (when (seq order-by)
    (let [[order-key order-direction] order-by
          direction                   (or order-direction :asc)]
      (format "%s %s"
              (csk/->PascalCaseString order-key)
              (str/upper-case (name direction))))))

(defn statement-builder
  ^StatementBuilder
  [params]
  (let [select       (select-expr params)
        from         (from-expr params)
        where        (where-expr params)
        order-by     (order-by-expr params)
        val-bindings (values-bindings params)]
    (cond-> (bind-statement-values val-bindings)
      select   (.select select)
      from     (.from from)
      where    (.where where)
      order-by (.orderBy order-by))))

;;; Client helpers

(def ^:private memoized->kebab-case-keyword
  (m/fifo (fn [input]
            (csk/->kebab-case-keyword input :separator #"(?<![A-Z])(?=[A-Z])"))
          {}
          :fifo/threshold 512))

(defn- transform-keys
  [response]
  (cske/transform-keys memoized->kebab-case-keyword response))

(defn- ensure-vecs
  [obj]
  (walk/postwalk (fn [node]
                   (if (seq? node)
                     (into [] node)
                     node))
                 obj))

(defn- convert
  [obj]
  (-> obj
      (j/from-java-deep {})
      (ensure-vecs)
      (transform-keys)))

(defn- lift-enums
  "Lift 'broken' enums after conversion from Java objects.

  SOAP API responses has many enums which after conversion from Java
  objects become Clojure maps with single `:value` key and string
  value.  This creates unnecessary noise and inconsistent with
  corresponding requests, where we just convert string values to
  appropriate classes.

  This function lifts those maps leaving just string values.

  Example:

  ```
  (lift-soap-enums {:foo {:value \"BAR\"}
                    :baz 123})
  ;; => {:foo \"BAR\", :baz 123}
  ```"
  [obj]
  (walk/postwalk (fn [node]
                   (if (and (map? node)
                            (contains? node :value)
                            (= 1 (count (keys node)))
                            (string? (:value node)))
                     (:value node)
                     node))
                 obj))

(defn execute-and-convert
  [request-method args]
  (try
    (when-let [response (apply request-method args)]
      (lift-enums (convert response)))
    (catch ApiException ^ApiException ex
      (throw (ex-info "SOAP API exception"
                      {:errors (into []
                                     (comp (map convert)
                                           (map lift-enums))
                                     (.getErrors ex))}
                      ex)))))

(defn execute-and-convert-paginated
  [fetch-page-fn page-size {:keys [limit vf]}]
  (let [xform (if limit
                (comp cat (take limit))
                cat)]
    (->> (iteration fetch-page-fn
                    :initk (int 0)
                    :kf (constantly page-size)
                    :vf (or vf identity))
         (into [] xform))))

;;; Date time helpers

(defn date-time
  ([]
   (date-time (Instant/now)))
  ([instant]
   (date-time instant (ZoneId/systemDefault)))
  ([instant time-zone-id]
   (let [time (ZonedDateTime/ofInstant instant time-zone-id)]
     {:date         {:day   (ZonedDateTime/.getDayOfMonth time)
                     :month (ZonedDateTime/.getMonthValue time)
                     :year  (ZonedDateTime/.getYear time)}
      :hour         (ZonedDateTime/.getHour time)
      :minute       (ZonedDateTime/.getMinute time)
      :second       (ZonedDateTime/.getSecond time)
      :time-zone-id (ZoneRegion/.toString (ZonedDateTime/.getZone time))})))
