(ns com.adgoji.ad-manager-orders.core
  (:require
   [com.adgoji.ad-manager-client.interface :as ad-manager-client]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.soap-client-utils.interface :as soap-client-utils])
  (:import
   (com.google.api.ads.admanager.axis.utils.v202502 StatementBuilder)
   (com.google.api.ads.admanager.axis.v202502 ApproveAndOverbookOrders ApproveOrders ApproveOrdersWithoutReservationChanges ArchiveOrders DeleteOrders DisapproveOrders DisapproveOrdersWithoutReservationChanges Order OrderPage OrderServiceInterface PauseOrders ResumeAndOverbookOrders ResumeOrders RetractOrders RetractOrdersWithoutReservationChanges SubmitOrdersForApproval SubmitOrdersForApprovalAndOverbook SubmitOrdersForApprovalWithoutReservationChanges UnarchiveOrders)))

(defn create
  [client orders]
  (let [service (ad-manager-client/service client :order)
        body    (->> orders
                     (sequence (map (partial java-utils/clj->java Order)))
                     (into-array Order))]
    (soap-client-utils/execute-and-convert OrderServiceInterface/.createOrders
                                           service
                                           body)))

(defn- fetch-orders-page
  [^OrderServiceInterface service ^StatementBuilder statement-builder offset]
  (.increaseOffsetBy statement-builder (int offset))
  (soap-client-utils/execute-and-convert (comp OrderPage/.getResults
                                               OrderServiceInterface/.getOrdersByStatement)
                                         service
                                         (.toStatement statement-builder)))

(defn all
  [client statement {:keys [limit]}]
  (let [service   (ad-manager-client/service client :order)
        page-size StatementBuilder/SUGGESTED_PAGE_LIMIT
        statement (-> (soap-client-utils/statement-builder statement)
                      (.limit page-size))]
    (soap-client-utils/execute-and-convert-paginated
     (partial fetch-orders-page service statement)
     page-size
     {:limit limit})))

(def ^:private order-action
  ;; TODO: Some actions accept an optional `skipInventoryCheck`
  ;; argument in the constructor.  Probably we should introduce some
  ;; function instead of a hash map.
  {:approve-orders                                         (ApproveOrders/new)
   :approve-and-overbook-orders                            (ApproveAndOverbookOrders/new)
   :approve-orders-without-reservation-changes             (ApproveOrdersWithoutReservationChanges/new)
   :archive-orders                                         (ArchiveOrders/new)
   :delete-orders                                          (DeleteOrders/new)
   :disapprove-orders                                      (DisapproveOrders/new)
   :disapprove-orders-without-reservation-changes          (DisapproveOrdersWithoutReservationChanges/new)
   :pause-orders                                           (PauseOrders/new)
   :resume-orders                                          (ResumeOrders/new)
   :resume-and-overbook-orders                             (ResumeAndOverbookOrders/new)
   :retract-orders                                         (RetractOrders/new)
   :retract-orders-without-reservation-changes             (RetractOrdersWithoutReservationChanges/new)
   :submit-orders-for-approval                             (SubmitOrdersForApproval/new)
   :submit-orders-for-approval-and-overbook                (SubmitOrdersForApprovalAndOverbook/new)
   :submit-orders-for-approval-without-reservation-changes (SubmitOrdersForApprovalWithoutReservationChanges/new)
   :unarchive-orders                                       (UnarchiveOrders/new)})

(defn perform-action
  [client order-id action]
  (let [service   (ad-manager-client/service client :order)
        ;; Potential future improvement: Support multiple Order IDs.
        ;; StatementBuilder doesn't allow binding a collection, so we
        ;; will need to generate multiple placeholders dynamically.
        statement (-> (soap-client-utils/statement-builder
                       {:where [[:= :id order-id]]})
                      (.toStatement))]
    (soap-client-utils/execute-and-convert OrderServiceInterface/.performOrderAction
                                           service
                                           (order-action action)
                                           statement)))
