(ns clj-http.lite.core
  "Core HTTP request/response implementation."
  (:require [clojure.java.io :as io])
  (:import (java.io ByteArrayOutputStream InputStream IOException)
           (java.net URI URL HttpURLConnection Socket)
           (javax.net.ssl SSLSocketFactory SSLSocket HttpsURLConnection)
           (clj-http.lite NoDHSocketFactory)))

(defn parse-headers
  "Takes a URLConnection and returns a map of names to values.

   If a name appears more than once (like `set-cookie`) then the value
   will be a vector containing the values in the order they appeared
   in the headers."
  [conn]
  (loop [i 1 headers {}]
    (let [k (.getHeaderFieldKey ^HttpURLConnection conn i)
          v (.getHeaderField ^HttpURLConnection conn i)]
      (if k
        (recur (inc i) (update-in headers [k] conj v))
        (zipmap (for [k (keys headers)]
                  (.toLowerCase ^String k))
                (for [v (vals headers)]
                  (if (= 1 (count v))
                    (first v)
                    (vec v))))))))

(defn- coerce-body-entity
  "Coerce the http-entity from an HttpResponse to either a byte-array, or a
  stream that closes itself and the connection manager when closed."
  [{:keys [as]} conn]
  (let [ins (try
              (.getInputStream ^HttpURLConnection conn)
              (catch Exception e
                (.getErrorStream ^HttpURLConnection conn)))]
    (if (= :stream as)
      ins
      (with-open [ins ^InputStream ins
                  baos (ByteArrayOutputStream.)]
        (io/copy ins baos)
        (.flush baos)
        (.toByteArray baos)))))

(defn- get-connection [^URL url]
  "Wrap .openConnection to "
  (let [conn (.openConnection url)]
    (if (instance? HttpsURLConnection conn)
      (doto conn
        (.setSSLSocketFactory
          (NoDHSocketFactory.
            (.getSSLSocketFactory conn))))
      conn)))

 
(defn request
  "Executes the HTTP request corresponding to the given Ring request map and
   returns the Ring response map corresponding to the resulting HTTP response.

   Note that where Ring uses InputStreams for the request and response bodies,
   the clj-http uses ByteArrays for the bodies."
  [{:keys [request-method scheme server-name server-port uri query-string
           headers content-type character-encoding body socket-timeout
           conn-timeout multipart debug insecure? save-request?] :as req}]
  (let [http-url (str (name scheme) "://" server-name
                      (when server-port (str ":" server-port))
                      uri
                      (when query-string (str "?" query-string)))
        conn (get-connection ^URL (URL. http-url))]
    (when (and content-type character-encoding)
      (.setRequestProperty conn "Content-Type" (str content-type
                                                    "; charset="
                                                    character-encoding)))
    (when (and content-type (not character-encoding))
      (.setRequestProperty conn "Content-Type" content-type))
    (doseq [[h v] headers]
      (.setRequestProperty conn h v))
    (.setRequestMethod ^HttpURLConnection conn (.toUpperCase (name request-method)))
    (when body
      (.setDoOutput conn true))
    (when socket-timeout
      (.setReadTimeout conn socket-timeout))
    (.connect conn)
    (when body
      (with-open [out (.getOutputStream conn)]
        (io/copy body out)))
    (merge {:headers (parse-headers conn)
            :status (.getResponseCode ^HttpURLConnection conn)
            :body (when-not (= request-method :head)
                    (coerce-body-entity req conn))}
           (when save-request?
             {:request (assoc (dissoc req :save-request?)
                         :http-url http-url)}))))
(comment

  (let [conn (.openConnection (URL. "https://addressbin.com/"))
        sf (.getSSLSocketFactory conn)
        nodh (NoDHSocketFactory. sf)
        ]
    (map prn (.getSupportedCipherSuites nodh)))

  (let [conn (get-connection (URL. "https://addressbin.com/"))]
    (map prn (.getSupportedCipherSuites (.getSSLSocketFactory conn))))

  (require '[clj-http.lite.client :as client])

  (def req {:accept :json,
            :as :json,
            :form-params {:grant_type "authorization_code",
                          :redirect_uri "http://www.redditlater.com/auth/callback",
                          :code "ePVsYwYZ6fH-sGM1K6LuA8N_3YQ"},
            :basic-auth ["-48Tu3HDoYk8zg" "ocuKyNPwNw6HP_6LioIhRSrx40E"]
   
            })
  (client/post "https://ssl.reddit.com/api/v1/access_token" req)

  (let [conn (.openConnection ^URL (URL. "https://addressbin.com/"))
        fac-before (.getSSLSocketFactory conn)
        fac-after (wrap-socket-factory fac-before)]

    (map prn (.getSupportedCipherSuites fac-before))
    (map prn (.getSupportedCipherSuites fac-after))
    )

)
