(ns aclaimant.server.core.maps
  (:require
    [clojure.string :as string]
    [clojure.walk :as walk]))

(defn ^:private filter-values-rec [func m]
  (walk/postwalk
    (fn [m2]
      (if (map? m2)
        (into (empty m2) (filter (comp func second) m2))
        m2))
    m))

(defn ^:private regex-replace [word regex replacement]
  (let [update #(string/replace % regex replacement)]
    (if (keyword? word)
      (keyword (update (name word)))
      (update word))))

(defn modify-key-values [aMap func]
  (walk/postwalk (fn [x]
              (if (map? x)
                (into {} (map func x))
                x))
            aMap))

(defn deep-merge
  "Recursively merges maps. If keys are not maps, the last value wins."
  [& vals]
  (if (every? map? vals)
    (apply merge-with deep-merge vals)
    (last vals)))

(defn modify-keys-recursive [aMap func]
  (let [f (fn [[k v]] [(func k) v])]
    (modify-key-values aMap f)))

(defn modify-vals-recursive [aMap func]
  (let [f (fn [[k v]] [k (func v)])]
    (modify-key-values aMap f)))

;; TODO: handle snake case too

(defn kebob-keys [aMap]
  (modify-keys-recursive aMap #(keyword (regex-replace (name %) #"_" "-"))))
(defn underscore-keys [aMap]
  (modify-keys-recursive aMap #(regex-replace % #"-" "_")))

(defn stringify-keys [aMap]
  (modify-keys-recursive aMap #(name %)))

(defn keywordize-keys [aMap]
  (modify-keys-recursive aMap #(keyword %)))

(defn dashify->snake [word]
  (let [finish-fn (if (keyword? word) keyword identity)]
    (-> word
        name
        (string/replace #"_" "-")
        (string/replace #"-(\w)" (fn [[_ m]] (string/upper-case m)))
        finish-fn)))

(defn jsonify-keys [aMap]
  (modify-keys-recursive aMap dashify->snake))

(defn snake->dashify-word [word]
  (let [finish-fn (if (keyword? word) keyword identity)]
    (-> word
        name
        (string/replace #"[A-Z]" (fn [m] (str "-" (string/lower-case m))))
        (string/replace #"^-" "")
        (string/replace #"_" "-")
        finish-fn)))

(defn dashify-keys
  ([aMap]
   (dashify-keys aMap {:keywordize? false}))
  ([aMap {:keys [keywordize?]}]
   (let [f (if keywordize?
             (comp keyword snake->dashify-word)
             snake->dashify-word)]
     (modify-keys-recursive aMap f))))

(defn without-nils-rec [obj]
  (filter-values-rec (complement nil?) obj))

(defn without-nils
  "Returns a map without any entries where the value is `nil?`"
  [m]
  (into {} (filter (comp not nil? val) m)))

(defn update-keys [m keys f]
  (reduce #(update %1 %2 f) m keys))

(defn map-of [m k-fn v-fn]
  (into {} (map (juxt k-fn v-fn) m)))

(defn select-by [k m]
  (into {}
        (for [[k [v]] (group-by k m)]
          [k v])))

(defn assoc-other [m m2 k]
  (assoc m k (k m2)))

(defn assoc-maybe [m k v]
  (if v (assoc m k v) m))

(defn update-maybe [m k f]
  (if (k m) (update m k f) m))
