(ns aclaimant.server.middleware
  (:require
    [aclaimant.server.core.request :as req]
    [aclaimant.server.core.content-type :as content-type]
    [aclaimant.dr-no.debug :as d]
    [cheshire.core :as json]
    [clojure.string :as string]
    [clojure.walk :as walk]
    [compojure.handler :as handler]
    [outpace.config :refer [defconfig]]
    [ring.middleware.json :as json-middleware]
    [ring.middleware.reload :as reload]))

(defconfig reload-middleware-enabled? false)

(def ^:private default-allow-headers
  ["Origin" "Content-Type" "Accept" "Authorization"])

(defn wrap-allow-cross-origin
  "middleware function to allow crosss origin - TODO restrict further"
  [handler & [{:keys [allow-headers]} :as args]]
  (let [allowed-headers (->> allow-headers (concat default-allow-headers) set)]
    (fn [request]
      (let [origin (get-in request [:headers "origin"])
            method (:request-method request)
            cors-headers {"Access-Control-Allow-Origin" origin
                          "Access-Control-Allow-Methods" "DELETE, GET, HEAD, PATCH, POST, PUT, OPTIONS"
                          "Access-Control-Allow-Credentials" "true"
                          "Access-Control-Allow-Headers" (string/join "," allowed-headers)}]
        (if (= :options method)
          {:status 200
           :headers cors-headers}
          (update (handler request) :headers merge cors-headers))))))

(defn wrap-json-errors
  ""
  [handler]
  (fn [request]
    (try
      (handler request)
      (catch Throwable e
        (println "[ERROR] an error occurred while processing a request."
                 (.getMessage e)
                 "\n"
                 (with-out-str (.printStackTrace e)))
        {:status 500
         :headers {"Content-Type" "application/json"}
         :body (json/generate-string
                 {:error (str (.getClass e))
                  :message (.getMessage e)})}))))

(defn wrap-json-request-params
  "Converts the snake-case keys in :params to dashified keys"
  [handler]
  (fn [req]
    (let [orig-params (:params req)
          orig-json-params (:json-params req)
          params (req/dashify-keys orig-params {:keywordize? true})
          json-params (req/dashify-keys orig-json-params {:keywordize? true})]
      (handler
        (assoc req
               :params (merge params json-params)
               :original-params orig-params
               :json-params json-params
               :original-json-params orig-json-params)))))

(defn wrap-logged [handler]
  (fn [{:keys [request-method uri headers] :as request}]
    (println (format "[REQUEST] %s %s" request-method uri))
    (let [start-time (System/currentTimeMillis)
          response (handler request)
          elapsed-millis (- (System/currentTimeMillis) start-time)]
      (println (format "[RESPONSE] %s %s %s %sms"
                       request-method
                       uri
                       (:status response)
                       elapsed-millis))
      response)))

(defn api-common
  ""
  [routes & [{:keys [content-type]}]]
  (-> routes
      ; TODO: Eventually (when firm is gone) assume all response bodies should
      ; be EDN unless a content-type is already set. No JSON. We'll need to make
      ; the EDN middleware handle org.joda.time.DateTime and org.bson.types.ObjectId
      (content-type/wrap-content-type-response content-type)
      wrap-json-request-params
      (json-middleware/wrap-json-body {:keywords? true :bigdecimals? true})
      json-middleware/wrap-json-params
      wrap-json-errors
      handler/api))

(defn older-api-common
  ""
  [routes & [{:keys []}]]
  (-> routes
      json-middleware/wrap-json-response
      (json-middleware/wrap-json-body {:keywords? true :bigdecimals? true})
      json-middleware/wrap-json-params
      wrap-json-errors
      handler/api))

(defn _id->id [response]
  (walk/postwalk
    (fn [m]
      (if (and (not (coll? m))
               (= (keyword m) :_id))
        :id
        m))
    response))

(defn ^:private body-fix-ids [body]
  (-> body
      json/parse-string
      _id->id
      json/generate-string))

(defn wrap-change-ids
  "Middleware that converts all :_id to :id
  in your response."
  [handler]
  (fn [request]
    (let [{:keys [headers] :as response} (handler request)]
      (if (= (get headers "Content-Type") "application/json")
        (update-in response [:body] body-fix-ids)
        response))))

(defn wrap-maybe-reload
  "Wrap with ring reload if configured using
  `{aclaimant.server.middleware.core/reload-middleware-enabled? true}`
  in your config.edn. Defaults to not reloading."
  [handler]
  (println "[Middleware] Wrap reload is set to" reload-middleware-enabled?)
  (if reload-middleware-enabled?
    (reload/wrap-reload handler)
    handler))
