(ns coconut.v1.running
  #?(:cljs (:require-macros [cljs.core.async.macros :as async]))
  (:require
    [coconut.v1.core :as core]
    [coconut.v1.platform :as platform]
    [coconut.v1.query :as query]
    [coconut.v1.matchers :as matchers]
    [clojure.core.async :as async :refer [<! >!]]
    ))

(defn ^{:private true} create-assertion-function
  ([assertion-event-atom time-provider]
   (fn assert-that
     ([actual matcher]
      (swap! assertion-event-atom
             conj
             (if-let [failure (matchers/evaluate-matcher matcher actual)]
               #::{:type ::assertion-failed
                   :expected (:expected failure)
                   :actual (:actual failure)
                   :current-time-millis (time-provider)}
               #::{:type ::assertion-passed
                   :current-time-millis (time-provider)}))))))

(defn ^{:private true} create-done-function
  ([done-channel]
   (fn done
     ([]
      (async/close! done-channel))
     ([e]
      (async/go
        (do (>! done-channel e)
            (async/close! done-channel)))))))

(defn ^{:private true} with-pending-information
  ([execution-state component]
   (if (nil? (::core/pending? component))
     execution-state
     (assoc execution-state
            ::pending? (::core/pending? component)
            ::pending-reason (::core/pending-reason component)))))

(defn ^{:private true} with-asynchronous-information
  ([execution-state component]
   (if (nil? (::core/asynchronous? component))
     execution-state
     (assoc execution-state
            ::asynchronous? (::core/asynchronous? component)
            ::timeout-in-milliseconds (::core/timeout-in-milliseconds component)))))

(defn ^{:private true} compose-around-each-functions
  ([f g]
   (fn [it]
     (f (fn x
          ([]
           (x (fn [])))
          ([a1]
           (g (fn y
                ([]
                 (y (fn [])))
                ([a2]
                 (it (fn []
                       (do (a2)
                           (a1)))))))))))))

(defn ^{:private true} with-around-each-components
  ([execution-state component]
   (update execution-state
           ::around-each
           #(transduce (map ::core/function)
                       (completing compose-around-each-functions)
                       %
                       (::core/components (::core/around-each component))))))

(defn ^{:private true} copy-to-channel
  ([channel collection]
   (let [close-channel? false]
     (async/onto-chan channel collection close-channel?))))

(defmulti ^{:private true} run-component
  (fn [event-channel context execution-state component]
    (::core/component-type component)))

(defmethod ^{:private true} run-component
  ::query/result
  ([event-channel context execution-state component]
   (async/go
     (<! (run-component event-channel
                        context
                        execution-state
                        (::core/sub-components component))))))

(defmethod ^{:private true} run-component
  ::core/collection
  ([event-channel context execution-state component]
   (async/go
     (doseq [component (::core/components component)]
       (<! (run-component event-channel
                          context
                          execution-state
                          component))))))

(defmethod ^{:private true} run-component
  ::core/before-all
  ([event-channel time-provider execution-state component]
   (async/go
     ((::core/function component)))))

(defmethod ^{:private true} run-component
  ::core/after-all
  ([event-channel time-provider execution-state component]
   (async/go
     ((::core/function component)))))

(defn synchronous->asynchronous
  ([component]
   (-> component
       (assoc ::core/asynchronous? true)
       (assoc ::core/timeout-in-milliseconds platform/max-int)
       (update ::core/function
               (fn [f]
                 (fn [assert-that done]
                   (do (f assert-that)
                       (done))))))))

(defmethod ^{:private true} run-component
  ::core/context
  ([event-channel time-provider execution-state component]
   (async/go
     (let [execution-state (-> execution-state
                               (with-pending-information component)
                               (with-asynchronous-information component)
                               (with-around-each-components component))]
       (<! (run-component event-channel
                          time-provider
                          execution-state
                          (::core/before-all component)))
       (>! event-channel
           #::{:type ::context-started
               :subject (::core/subject component)
               :current-time-millis (time-provider)})
       (<! (run-component event-channel
                          time-provider
                          execution-state
                          (::core/sub-components component)))
       (>! event-channel
           #::{:type ::context-finished
               :subject (::core/subject component)
               :current-time-millis (time-provider)})
       (<! (run-component event-channel
                          time-provider
                          execution-state
                          (::core/after-all component)))))))

(defmethod ^{:private true} run-component
  ::core/test
  ([event-channel time-provider original-execution-state component]
   (let [execution-state (-> original-execution-state
                             (with-pending-information component)
                             (with-asynchronous-information component))]
     (async/go
       (cond (::pending? execution-state)
             (>! event-channel
                 #::{:type ::test-marked-pending
                     :definition-line-number (::core/definition-line-number component)
                     :description (::core/description component)
                     :pending-reason (::pending-reason execution-state)
                     :current-time-millis (time-provider)})

             (::asynchronous? execution-state)
             (let [around-each (::around-each execution-state)
                   execution-finished-channel (async/chan)]
               (around-each (fn execute
                              ([]
                               (execute (fn [])))
                              ([after-function]
                               (let [assertion-event-atom (atom [])
                                     done-channel (async/chan)
                                     timeout-channel (async/timeout (::timeout-in-milliseconds execution-state))
                                     test-function (::core/function component)
                                     assertion-function (create-assertion-function assertion-event-atom time-provider)
                                     done-function (create-done-function done-channel)]
                                 (async/go
                                   (>! event-channel
                                       #::{:type ::test-started
                                           :namespace-name (::core/namespace-name component)
                                           :definition-line-number (::core/definition-line-number component)
                                           :description (::core/description component)
                                           :current-time-millis (time-provider)})
                                   (try (test-function assertion-function done-function)
                                        (catch #?(:clj Throwable :cljs :default) e
                                          (done-function e)))
                                   (let [[completion-value completion-channel] (async/alts! [done-channel timeout-channel])]
                                     (<! (copy-to-channel event-channel @assertion-event-atom))
                                     (cond (= timeout-channel completion-channel)
                                           (>! event-channel
                                               #::{:type ::test-timed-out
                                                   :description (::core/description component)
                                                   :current-time-millis (time-provider)})

                                           (platform/exception? completion-value)
                                           (>! event-channel
                                               #::{:type ::test-threw-exception
                                                   :description (::core/description component)
                                                   :exception completion-value
                                                   :current-time-millis (time-provider)})

                                           :else
                                           (>! event-channel
                                               #::{:type ::test-finished
                                                   :description (::core/description component)
                                                   :current-time-millis (time-provider)}))
                                     (after-function)
                                     (async/close! execution-finished-channel)))))))
               (<! execution-finished-channel))

             :else
             (<! (run-component event-channel
                                time-provider
                                original-execution-state
                                (synchronous->asynchronous component))))))))

(defn run
  "Given a query result, runs each component and returns a core.async channel
  containing the events produced. The events only represent what happened when
  the components were run. No assumptions are made about what constitutes a
  passing or failing test, context, or suite."
  ([query-result]
   (run query-result platform/current-time-millis))
  ([query-result time-provider]
   (let [event-channel (async/chan 250)
         execution-state #::{:asynchronous? false
                             :timeout-in-milliseconds nil
                             :pending? false
                             :pending-reason nil
                             :around-each #(%)}]
     (async/go
       (>! event-channel
           #::{:type ::suite-started
               :current-time-millis (time-provider)
               :total-number-of-tests (::query/total-number-of-tests query-result)})
       (<! (run-component event-channel
                          time-provider
                          execution-state
                          query-result))
       (>! event-channel
           #::{:type ::suite-finished
               :current-time-millis (time-provider)})
       (async/close! event-channel))
     event-channel)))
