(ns coconut.v1.results
  #?(:cljs (:require-macros [cljs.core.async.macros :as async]))
  (:require
    [coconut.v1.platform :as platform]
    [coconut.v1.summarizing :as summarizing] [coconut.v1.rendering :as rendering]
    [coconut.v1.aggregation :as aggregation]
    [clojure.core.async :as async :refer [<! >!]]
    ))

(defn ^{:private true} with-indicies
  ([elements]
   (into (vector)
         (comp (map-indexed vector)
               (map #(update % 0 inc)))
         elements)))

(def ^{:private true :const true} checkmark \✔)
(def ^{:private true :const true} x \✘)
(def ^{:private true :const true} asterisk \*)

(defmulti report-event
  (fn [state event]
    (::summarizing/type event)))

(defmethod report-event
  ::summarizing/suite-started
  ([state event]
   #::aggregation{:events []
                  :state #::{:total-number-of-tests (::summarizing/total-number-of-tests event)
                             :pending-tests []
                             :failing-tests []
                             :level 0}}))

(defmethod report-event
  ::summarizing/context-started
  ([state event]
   #::aggregation{:events []
                  :state (update state
                                 ::level inc)}))

(defmethod report-event
  ::summarizing/context-finished
  ([state event]
   #::aggregation{:events []
                  :state (update state
                                 ::level dec)}))

(defmethod report-event
  ::summarizing/test-pending
  ([state event]
   #::aggregation{:events []
                  :state (update state
                                 ::pending-tests conj event)}))

(defmethod report-event
  ::summarizing/test-passed
  ([state event]
   #::aggregation{:events []
                  :state state}))

(defmethod report-event
  ::summarizing/test-failed
  ([state event]
   #::aggregation{:events []
                  :state (update state
                                 ::failing-tests conj event)}))

(defmethod report-event
  ::summarizing/test-threw-exception
  ([state event]
   #::aggregation{:events []
                  :state (update state
                                 ::failing-tests conj event)}))

(defmethod report-event
  ::summarizing/test-timed-out
  ([state event]
   #::aggregation{:events []
                  :state (update state
                                 ::failing-tests conj event)}))

(defmethod report-event
  ::summarizing/suite-finished
  ([state event]
   (let [put! (fn [a e]
                (swap! a conj e))
         events (atom [])]
     (when (seq (::pending-tests state))
       (put! events
             #::rendering{:type ::rendering/newline})
       (put! events
             #::rendering{:type ::rendering/line
                          :segments [#::rendering{:text "Pending..."}]})
       (doseq [[n t] (with-indicies (::pending-tests state))]
         (let [list-index (rendering/render-list-index n)]
           (put! events
                 #::rendering{:type ::rendering/newline})
           (let [description-lines (rendering/render-full-description (::summarizing/context t)
                                                                      (::summarizing/description t))]
             (put! events
                   #::rendering{:type ::rendering/line
                                :segments [#::rendering{:text (rendering/indent 1)}
                                           #::rendering{:text list-index}
                                           #::rendering{:text (rendering/spacing 1)}
                                           #::rendering{:text (first description-lines)}]})
             (doseq [line (rest description-lines)]
               (put! events
                     #::rendering{:type ::rendering/line
                                  :segments [#::rendering{:text (rendering/indent 1)}
                                             #::rendering{:text (rendering/spacing (count list-index))}
                                             #::rendering{:text (rendering/spacing 1)}
                                             #::rendering{:text line}]})))
           (put! events
                 #::rendering{:type ::rendering/line
                              :segments [#::rendering{:text (rendering/indent 1)}
                                         #::rendering{:text (rendering/spacing (count list-index))}
                                         #::rendering{:text (rendering/spacing 1)}
                                         #::rendering{:text (rendering/pending-reason (::summarizing/pending-reason t))
                                                      :color ::rendering/yellow}]}))))

     (when (seq (::failing-tests state))
       (put! events
             #::rendering{:type ::rendering/newline})
       (put! events
             #::rendering{:type ::rendering/line
                          :segments [#::rendering{:text "Failures..."}]})
       (doseq [[n t] (with-indicies (::failing-tests state))]
         (let [list-index (rendering/render-list-index n)]
           (put! events
                 #::rendering{:type ::rendering/newline})
           (let [description-lines (rendering/render-full-description (::summarizing/context t)
                                                                      (::summarizing/description t))]
             (put! events
                   #::rendering{:type ::rendering/line
                                :segments [#::rendering{:text (rendering/indent 1)}
                                           #::rendering{:text list-index}
                                           #::rendering{:text (rendering/spacing 1)}
                                           #::rendering{:text (first description-lines)}]})
             (doseq [line (rest description-lines)]
               (put! events
                     #::rendering{:type ::rendering/line
                                  :segments [#::rendering{:text (rendering/indent 1)}
                                             #::rendering{:text (rendering/spacing (count list-index))}
                                             #::rendering{:text (rendering/spacing 1)}
                                             #::rendering{:text line}]})))
           (put! events
                 #::rendering{:type ::rendering/line
                              :segments [#::rendering{:text (rendering/indent 1)}
                                         #::rendering{:text (rendering/spacing (count list-index))}
                                         #::rendering{:text (rendering/spacing 2)}
                                         #::rendering{:text "namespace:"
                                                      :color ::rendering/teal}
                                         #::rendering{:text (rendering/spacing 1)}
                                         #::rendering{:text (::summarizing/namespace-name t)
                                                      :color ::rendering/red}]})
           (put! events
                 #::rendering{:type ::rendering/line
                              :segments [
                                         #::rendering{:text (rendering/indent 1)}
                                         #::rendering{:text (rendering/spacing (count list-index))}
                                         #::rendering{:text (rendering/spacing 7)}
                                         #::rendering{:text "line:"
                                                      :color ::rendering/teal}
                                         #::rendering{:text (rendering/spacing 1)}
                                         #::rendering{:text (str (::summarizing/definition-line-number t))
                                                      :color ::rendering/red}]})
           (case (::summarizing/type t)
             ::summarizing/test-failed
             (do (put! events
                       #::rendering{:type ::rendering/line
                                    :segments [#::rendering{:text (rendering/indent 1)}
                                               #::rendering{:text (rendering/spacing (count list-index))}
                                               #::rendering{:text (rendering/spacing 3)}
                                               #::rendering{:text "expected:"
                                                            :color ::rendering/teal}
                                               #::rendering{:text (rendering/spacing 1)}
                                               #::rendering{:text (first (::summarizing/expected t))
                                                            :color ::rendering/red}]})
                 (doseq [line (rest (::summarizing/expected t))]
                   (put! events
                         #::rendering{:type ::rendering/line
                                      :segments [#::rendering{:text (rendering/indent 1)}
                                                 #::rendering{:text (rendering/spacing (count list-index))}
                                                 #::rendering{:text (rendering/spacing 13)}
                                                 #::rendering{:text line
                                                              :color ::rendering/red}]}))
                 (when-let [actual (::summarizing/actual t)]
                   (put! events
                         #::rendering{:type ::rendering/line
                                      :segments [#::rendering{:text (rendering/indent 1)}
                                                 #::rendering{:text (rendering/spacing (count list-index))}
                                                 #::rendering{:text (rendering/spacing 5)}
                                                 #::rendering{:text "actual:"
                                                              :color ::rendering/teal}
                                                 #::rendering{:text (rendering/spacing 1)}
                                                 #::rendering{:text (first (::summarizing/actual t))
                                                              :color ::rendering/red}]})
                   (doseq [line (rest (::summarizing/actual t))]
                     (put! events
                           #::rendering{:type ::rendering/line
                                        :segments [#::rendering{:text (rendering/indent 1)}
                                                   #::rendering{:text (rendering/spacing (count list-index))}
                                                   #::rendering{:text (rendering/spacing 13)}
                                                   #::rendering{:text line
                                                                :color ::rendering/red}]}))))

             ::summarizing/test-threw-exception
             (let [e (::summarizing/exception t)
                   stack-trace (platform/exception-stack-trace e)]
               (put! events
                     #::rendering{:type ::rendering/line
                                  :segments [#::rendering{:text (rendering/indent 1)}
                                             #::rendering{:text (rendering/spacing (count list-index))}
                                             #::rendering{:text (rendering/spacing 2)}
                                             #::rendering{:text "exception:"
                                                          :color ::rendering/teal}
                                             #::rendering{:text (rendering/spacing 1)}
                                             #::rendering{:text (platform/exception-class-name e)
                                                          :color ::rendering/red}]})
               (put! events
                     #::rendering{:type ::rendering/line
                                  :segments [#::rendering{:text (rendering/indent 1)}
                                             #::rendering{:text (rendering/spacing (count list-index))}
                                             #::rendering{:text (rendering/spacing 4)}
                                             #::rendering{:text "message:"
                                                          :color ::rendering/teal}
                                             #::rendering{:text (rendering/spacing 1)}
                                             #::rendering{:text (or (platform/exception-message e) "-")
                                                          :color ::rendering/red}]})
               (put! events
                     #::rendering{:type ::rendering/line
                                  :segments [#::rendering{:text (rendering/indent 1)}
                                             #::rendering{:text (rendering/spacing (count list-index))}
                                             #::rendering{:text "stack trace:"
                                                          :color ::rendering/teal}
                                             #::rendering{:text (rendering/spacing 1)}
                                             #::rendering{:text (first stack-trace)
                                                          :color (rendering/stack-trace-element-color (first stack-trace))}]})
               (doseq [stack-trace-element (rest stack-trace)]
                 (put! events
                       #::rendering{:type ::rendering/line
                                    :segments [#::rendering{:text (rendering/indent 1)}
                                               #::rendering{:text (rendering/spacing (count list-index))}
                                               #::rendering{:text (rendering/spacing 13)}
                                               #::rendering{:text stack-trace-element
                                                            :color (rendering/stack-trace-element-color stack-trace-element)}]})))

             ::summarizing/test-timed-out
             (do (put! events
                       #::rendering{:type ::rendering/line
                                    :segments [#::rendering{:text (rendering/indent 1)}
                                               #::rendering{:text (rendering/spacing (count list-index))}
                                               #::rendering{:text (rendering/spacing 5)}
                                               #::rendering{:text "reason:"
                                                            :color ::rendering/teal}
                                               #::rendering{:text (rendering/spacing 1)}
                                               #::rendering{:text "TIMEOUT"
                                                            :color ::rendering/red}]}))

             (throw (platform/illegal-argument-exception
                      (str "unsupported summarizing event type: "
                           (::summarizing/type t))))))))

     (put! events
           #::rendering{:type ::rendering/newline})
     (put! events
           #::rendering{:type ::rendering/line
                        :segments [#::rendering{:text (platform/format "Finished in %s seconds"
                                                                       (rendering/render-suite-duration (::summarizing/duration-in-milliseconds event)))}]})
     (put! events
           #::rendering{:type ::rendering/line
                        :segments [#::rendering{:text (rendering/render-number-of-tests (::total-number-of-tests state)
                                                                                        (count (::failing-tests state))
                                                                                        (count (::pending-tests state)))
                                                :color (rendering/number-of-tests-color (count (::failing-tests state))
                                                                                        (count (::pending-tests state)))}]})
     #::aggregation{:events @events
                    :state state})))

(defmethod report-event
  :default
  ([state event]
   #::aggregation{:events []
                  :state state}))
