(ns coconut.v1.matchers
  #?(:cljs (:require-macros [coconut.v1.matchers]))
  (:require
    [clojure.set :as cs]
    [coconut.v1.platform :as platform]
    [clojure.pprint :as pp]
    [clojure.string :as cstr]
    ))

(defn render
  ([form dispatch]
   (-> form
       (pp/write :dispatch dispatch
                 :pretty true)
       (with-out-str)
       (cstr/split #"\n"))))

(defn render-data
  ([form]
   (render form
           pp/simple-dispatch)))

(defn render-code
  ([form]
   (render form
           pp/code-dispatch)))

(defn multi-or-single-line-message
  ([options]
   (let [render-fn (::render-fn options render-data)
         rendered-data (render-fn (::data options))]
     (if (< 1 (count rendered-data))
       (list* (str (::message options)
                   \…)
              rendered-data)
       (list (apply str
                    (::message options)
                    \space
                    rendered-data))))))

(defprotocol IMatcher
  (evaluate-matcher [matcher actual]))

(extend-protocol IMatcher
  #?(:clj clojure.lang.IFn
     :cljs function)
  (evaluate-matcher [matcher actual]
    (matcher actual)))

(defn create-equality-matcher
  ([expected]
   (fn [actual]
     (when-not (= expected actual)
       {:expected (render-data expected)
        :actual (render-data actual)}))))

(defn create-negated-equality-matcher
  ([expected]
   (fn [actual]
     (when (= expected actual)
       {:expected (multi-or-single-line-message
                    #::{:message "something other than"
                        :data expected})}))))

(defn create-collection-membership-matcher
  ([value]
   (fn [collection]
     (when-not (contains? (set collection) value)
       {:expected (multi-or-single-line-message
                    #::{:message "a collection containing"
                        :data value})
        :actual (render-data collection)}))))

(defn create-subset-matcher
  ([elements]
   (fn [coll]
     (when-not (cs/subset? (set elements) (set coll))
       {:expected (multi-or-single-line-message
                    #::{:message "a collection containing"
                        :data elements})
        :actual (render-data coll)}))))

#?(:clj
(defmacro create-satisfies-predicate-matcher
  ([predicate]
   `(fn [value#]
      (when-not (~predicate value#)
        {:expected (multi-or-single-line-message
                     #::{:message "a value which satisfies"
                         :data '~predicate
                         :render-fn render-code})
         :actual (render-data value#)})))))

(defn create-throws-exception-matcher
  ([]
   (create-throws-exception-matcher platform/throwable-type))
  ([exception-class]
   (create-throws-exception-matcher exception-class nil))
  ([exception-class exception-message]
   (fn [f]
     (try (f)
          {:expected [(str "a " (platform/exception-class-name exception-class) " to be thrown")]
           :actual ["no exception"]}
          (catch #?(:clj Throwable :cljs :default) t
            (if-not (instance? exception-class t)
              {:expected [(str "a " (platform/exception-class-name exception-class) " to be thrown")]
               :actual (render-data t)}
              (when exception-message
                (when (not= (platform/exception-message t) exception-message)
                  {:expected [(str "an exception message of " (pr-str exception-message))]
                   :actual (render-data t)}))))))))

(defn create-regex-matcher
  ([regex]
   (fn [string]
     (when-not (and (string? string)
                    (re-matches regex string))
       {:expected (multi-or-single-line-message
                    #::{:message "a string matching"
                        :data (render-data regex)})
        :actual (render-data string)}))))
