(ns coconut.v1.main
  (:require
    #?(:clj [clojure.core.async :as async])
    #?(:cljs [coconut.v1.karma :as karma])
    #?(:cljs [coconut.v1.html :as html])
    #?(:cljs [coconut.v1.console :as console])
    [coconut.v1.platform :as platform]
    [coconut.v1.query :as query]
    [coconut.v1.running :as running]
    [coconut.v1.summarizing :as summarizing]
    [coconut.v1.documentation :as documentation]
    [coconut.v1.progress :as progress]
    [coconut.v1.results :as results]
    [coconut.v1.cli :as cli]
    [coconut.v1.util :as util]
    [coconut.v1.aggregation :as aggregation]
    ))

(defn reporter
  ([sym]
   (case sym
     :documentation documentation/report-event
     :progress progress/report-event
     :results results/report-event
     (throw (platform/illegal-argument-exception
              (platform/format "unsupported reporter: %s" (pr-str sym)))))))

(defn reporters
  ([options]
   (into (vector)
         (map reporter)
         (:reporters options))))

(defn output
  ([options]
   (case (:output options)
     :cli cli/output
     #?@(:cljs [:html html/output])
     #?@(:cljs [:browser-console console/output])
     (throw (platform/illegal-argument-exception
              (platform/format "unsupported output: %s"
                               (:output options)))))))

#?(:clj
(defn normalize-options
  ([options] options)))

#?(:cljs
(defn normalize-options
  ([options]
   (-> options
       (js->clj :keywordize-keys true)
       (update :reporters (partial map keyword))
       (update :output keyword)))))

#?(:clj
(defn complete
  ([c]
   (async/<!! c))))

#?(:cljs
(defn complete
  ([c])))

(defn ^{:export true} run
  "Runs the test suite according to the given options. Options should
  be a map containing the following keys:

  :reporters
  A vector of reporters to use. Possible options are...
    :progress         Renders single-character results for a test similar
                      to RSpec.

    :documentation    Renders the test description for each test as it
                      finishes similar to RSpec.

    :results          Renders the final report of a test run as well as
                      infomration about failing tests, pending tests, tests
                      which threw an exception, and asynchronous tests which
                      timed out.

  :output
  Where to output the rendered test reports. Possible options are...
    :cli             Prints the rendered test report data to the terminal.

    :html            Appends styled HTML elements to the DOM. (ClojureScript only).

    :browser-console Outputs colored browser console messages. (ClojureScript only).

  :criteria
  The query used to determine which tests to run. Possible options are...
    :all                 Matches any defined test

    :within-namespace    Matches any test defined within the given namespace.
                        \"Namespace\" here refers to one of three possible things...
                         1.) The string name of the namespace.
                         2.) The symbol name of the namespace.
                         2.) The actual namespace object (Clojure only).

    :within-namespaces   Matches any test defined in any of the given namespaces.
                         See above for the possible types of the namespace arguments.

    :not                 Negates the given criteria.

    :or                  Combines two criteria using logical OR.

    :and                 Combines two criteria using logical AND.

    :defined-on-line     Returns any test which is defined on a specific line.
                         Typically this will be used in conjection with something like
                         :within-namespace to run a certain test.

    :description-matches Matches any test whose full description matches the given
                         regex. The description will be the full test description
                         which is the test defined on the text itself prepended with
                         the text of each context within which it is defined.

    :has-tag             Matches any test which contains the given tag. Tags are matched
                         using the the standard #'clojure.core/=.

  Example:
  (run {:reporters [:progress :results]
        :output :cli
        :criteria [:and
                   [:namespace-matches #\"foo.*-test\"]
                   [:not [:or [:has-tag :integration]
                              [:has-tag :slow]]]]})

  (run {:reporters [:documentation :results]
        :output :html
        :criteria [:all]})"
  ([options]
   (let [options (normalize-options options)
         reporter (->> (reporters options)
                       (aggregation/compose)
                       (aggregation/aggregation))
         output (output options)]
     (-> (:criteria options)
         (query/query)
         (running/run)
         (summarizing/summarize)
         (reporter)
         (output)
         (complete)))))
