(ns coconut.v1.core
  (:refer-clojure :exclude [let for])
  #?(:cljs (:require-macros [coconut.v1.core]))
  (:require
    [coconut.v1.matchers :as matchers]
    [coconut.v1.platform :as platform]
    ))

(defn ^{:private true} asynchronous?
  ([options]
   (when (contains? options :asynchronous)
     (boolean (:asynchronous options)))))

(defn ^{:private true} timeout-in-milliseconds
  ([options]
   (:timeout (:asynchronous options))))

(defn ^{:private true} pending?
  ([options]
   (when (contains? options :pending)
     (boolean (:pending options)))))

(defn ^{:private true} pending-reason
  ([options]
   (when (string? (:pending options))
     (:pending options))))

(defn ^{:private true} tags
  ([options]
   (set (:tags options))))

(defn ^{:private true} with-denormalized-options
  ([component options]
   (merge component
          options
          #::{:asynchronous? (asynchronous? options)
              :timeout-in-milliseconds (timeout-in-milliseconds options)
              :pending? (pending? options)
              :pending-reason (pending-reason options)
              :tags (tags options)})))

(defn ^{:private true} append-components-to-collection
  ([collection-component components]
   (update collection-component ::components concat components)))

(declare append-component-to-collection)

(defn ^{:private true} add-sub-component
  ([context component]
   (clojure.core/let [component-type (::component-type component)]
     (cond (or (= ::context component-type)
               (= ::test component-type))
           (update context
                   ::sub-components
                   append-component-to-collection
                   component)

           (= ::collection component-type)
           (add-sub-component context
                               (::components component))

           (or (= ::before-all component-type)
               (= ::after-all component-type)
               (= ::around-each component-type))
           (update context
                   component-type
                   append-component-to-collection
                   component)

           (sequential? component)
           (reduce add-sub-component
                   context
                   component)

           :else
           (throw (platform/illegal-argument-exception
                    (str "unsupported component: " (pr-str component))))))))

(defn ^{:private true} add-sub-components
  ([context components]
   (reduce add-sub-component context components)))

(declare component?)

(defn ^{:private true} user-defined-component?
  ([object]
   (or (component? object)
       (sequential? object))))

(defn ^{:private true} options-and-components
  ([arguments]
   (as-> arguments __
         (remove var? __)
         (remove nil? __)
         (split-with (complement user-defined-component?) __)
         (update __ 0 (partial reduce merge)))))

(defn component?
  "Returns true when the given object is a coconut component. Interna
  implementation detail and subject to change."
  ([object]
   (boolean (::component-type object))))

(defn string-label
  "Returns the label for a component as a string. Returns the subject
  for a context or the description for a test component. Implementation
  detail and subject to change."
  ([component]
   (clojure.core/let [value (case (::component-type component)
                              ::context (::subject component)
                              ::test (::description component))]
     (if (string? value)
       value
       (pr-str value)))))

(defn append-component-to-collection
  "Appends a component to a collection component. Implementation detail
  and subject to change."
  ([collection-component component]
   (append-components-to-collection collection-component [component])))

(defn prepend-component-to-collection
  "Prepends a component to a collection component. Implementation detail
  and subject to change."
  ([collection-component component]
   (update collection-component ::components (partial into [component]))))

(defn create-collection-component
  "Low-level function for creating a collection component. Prefer using
  an object which satisfies #'clojure.core/sequential? such as a list
  or a vector."
  ([]
   (create-collection-component []))
  ([components]
   #::{:component-type ::collection
       :components components}))

(defn create-context-component
  "Low-level function to create a context component. Prefer using
  the #'coconut.v1.core/context macro."
  ([file-name namespace-name definition-line-number subject & arguments]
   (clojure.core/let [[options components] (options-and-components arguments)]
     (-> #::{:component-type ::context
             :id (platform/generate-uuid)
             :file-name file-name
             :namespace-name namespace-name
             :definition-line-number definition-line-number
             :subject subject
             :before-all (create-collection-component)
             :after-all (create-collection-component)
             :around-each (create-collection-component)
             :sub-components (create-collection-component)}
         (with-denormalized-options options)
         (add-sub-components components)))))

(defn create-around-each-component
  "Low-level function to create an around each component. Prefer using
  the #'coconut.v1.core/around-each macro."
  ([f]
   #::{:component-type ::around-each
       :function f}))

(defn create-before-all-component
  "Low-level function to create a before all component. Prefer using
  the #'coconut.v1.core/before-all macro."
  ([f]
   #::{:component-type ::before-all
       :function f}))

(defn create-after-all-component
  "Low-level function to create an after all component. Prefer using
  the #'coconut.v1.core/after-all component."
  ([f]
   #::{:component-type ::after-all
       :function f}))

(defn create-before-each-component
  "Low-level function to create a before each component. Prefer using
  the #'coconut.v1.core/before-each macro."
  ([f]
   (create-around-each-component
     (fn [it]
       (do (f)
           (it))))))

(defn create-after-each-component
  "Low-level function to create an after each component. Prefer using
  the #'coconut.v1.core/after-each macro."
  ([f]
   (create-around-each-component
     (fn [it]
       (it f)))))

(defn create-test-component
  "Low-level function to create a test component. Prefer using the
  #'coconut.v1.core/it macro to define a test within a context or #'coconut.v1.core/deftest
  to define a top-level test."
  ([file-name namespace-name definition-line-number description function]
   (create-test-component file-name namespace-name definition-line-number description {} function))
  ([file-name namespace-name definition-line-number description options function]
   (-> #::{:component-type ::test
           :id (platform/generate-uuid)
           :file-name file-name
           :namespace-name namespace-name
           :definition-line-number definition-line-number
           :description description
           :function function}
       (with-denormalized-options options))))

#?(:clj
(defmacro is
  "Verifies the expected and actual values are equal.
  Compared using #'clojure.core/=.

  Example:
  (assert-that 42 (is 42))"
  ([& arguments]
   `(matchers/create-equality-matcher ~@arguments))))

#?(:clj
(defmacro is-not
  "Verifies the expected and actual values are not equal.
  Compared using #'clojure.core/=.

  Example:
  (assert-that 42 (is 42))"
  ([& arguments]
   `(matchers/create-negated-equality-matcher ~@arguments))))

#?(:clj
(defmacro contains-value
  "Verifies the collection contains the expected value. Collections
  are coerced into a set.

  Example:
  (assert-that [42] (contains-value 42))"
  ([& arguments]
   `(matchers/create-collection-membership-matcher ~@arguments))))

#?(:clj
(defmacro contains
  "Verifies the provided collection contains all expected values.
  The collection and the sequence of expected items are compared
  using #'clojure.set/subset?.

  Example:
  (assert-that [1 2 3] (contains [2]))"
  ([& arguments]
   `(matchers/create-subset-matcher ~@arguments))))

#?(:clj
(defmacro satisfies
  "Verifies the value satisifes the given predicate.

  Example:
  (assert-that 42 (satisfies number?))"
  ([& arguments]
   `(matchers/create-satisfies-predicate-matcher ~@arguments))))

#?(:clj
(defmacro throws
  "When invoked with no arguments verifies the given functions throws
  any type of exception. When a particular exception class is given,
  verifies the given function throws an exception of that type or any
  subtype. Finally, when invoked with a class and a message, verifies
  an exception of the correct type with the expected message is thrown.
  Comparison of exception class is done using #'clojure.core/instance?.

  Example:
  (let [f #(throw (Exception. \"ouch\"))]
    (assert-that f (throws))
    (assert-that f (throws Exception))
    (assert-that f (throws Exception \"ouch\")))"
  ([& arguments]
   `(matchers/create-throws-exception-matcher ~@arguments))))

#?(:clj
(defmacro matches
  "Verifies a string matches the given regular expression."
  ([& arguments]
   `(matchers/create-regex-matcher ~@arguments))))

#?(:clj
(defmacro let
  "The same as #'clojure.core/let but lets multiple tests be defined."
  ([bindings & components]
   `(clojure.core/let [~@bindings] [~@components]))))

#?(:clj
(defmacro for
  "The same as #'clojure.core/for but lets multiple tests be defined."
  ([bindings & components]
   `(clojure.core/for [~@bindings] [~@components]))))

#?(:clj
(defmacro it
  "Creates a test. Tests take a description of what is currently
  being tested, any number of key/value options, and a test
  function. When tests are synchronous, test functions should accept
  a single argument--a function for making assertions. If the test
  is asynchronous, the test function should accept an assertion
  function as well as a function for notifying the test framework
  when the test is complete.

  Examples:
  (it \"is synchronous\"
    (fn [assert-that]
      (assert-that 42 (is 42))))

  (it \"is asynchronous\"
    :asynchronous {:timeout 250}
    (fn [assert-that done]
      (assert-that 42 (is 42))
      (done)))"
  ([& arguments]
   (clojure.core/let [file-name *file*
                      namespace-name (str *ns*)
                      definition-line-number (:line (meta &form))]
     `(create-test-component
        ~file-name
        ~namespace-name
        ~definition-line-number
        ~@arguments)))))

#?(:clj
(defmacro context
  "Creates a test context. Contexts can be used to group tests which
  have similar setup, test similar aspects of an entity.

  Example:
  (context \"foo\"
    (it \"is related to foo\"
      (fn [assert-that] ...))

    (it \"is also related to foo\"
      (fn [assert-that] ...)))"
  ([& arguments]
   (clojure.core/let [file-name *file*
                      namespace-name (str *ns*)
                      definition-line-number (:line (meta &form))]
     `(create-context-component
        ~file-name
        ~namespace-name
        ~definition-line-number
        ~@arguments)))))

#?(:clj
(defmacro before-all
  "Runs the given zero-arity function before all tests in the
  containing context.

  Example:
  (context \"foo\"
    (before-all (fn [] ...))

    (it \"has the before all function run before it\"
      (fn [assert-that]
        ...)))"
  ([& arguments]
   `(create-before-all-component ~@arguments))))

#?(:clj
(defmacro before-each
  "Runs the given zero-arity function before each test in the
  containing context.

  Example:
  (context \"foo\"
    (before-each (fn [] ...))

    (it \"has the before each function run before it\"
      (fn [assert-that]
        ...)))"
  ([& arguments]
   `(create-before-each-component ~@arguments))))

#?(:clj
(defmacro after-each
  "Runs the given zero-arity function after each test in the
  containing context.

  Example:
  (context \"foo\"
    (after-each (fn [] ...))

    (it \"has the after each function run after it\"
      (fn [assert-that]
        ...)))"
  ([& arguments]
   `(create-after-each-component ~@arguments))))

#?(:clj
(defmacro after-all
  "Runs the given zero-arity function after all tests in the
  containing context.

  Example:
  (context \"foo\"
    (after-all (fn [] ...))

    (it \"has the after all function run after it\"
      (fn [assert-that]
        ...)))"
  ([& arguments]
   `(create-after-all-component ~@arguments))))

#?(:clj
(defmacro around-each
  ([& arguments]
   `(create-around-each-component ~@arguments))))

#?(:clj
(defmacro describe
  "Defines a collection of tests.

  Example:
  (describe #'core/foo
    (it \"does something\"
      (fn [assert-that] ...)))"
  ([& arguments]
   (clojure.core/let [file-name *file*
                      namespace-name (str *ns*)
                      definition-line-number (:line (meta &form))]
     `(clojure.core/let [component# (create-context-component
                                      ~file-name
                                      ~namespace-name
                                      ~definition-line-number
                                      ~@arguments)]
        (platform/register-component component#
                                     ~namespace-name
                                     {::platform/component-version 1}))))))

#?(:clj
(defmacro deftest
  "Defines a single test.

  Example:
  (deftest \"+ returns the sum of two numbers\"
    (fn [assert-that]
      (assert-that (+ 1 2) (c/is 3)))))"
  ([& arguments]
   (clojure.core/let [file-name *file*
                      namespace-name (str *ns*)
                      definition-line-number (:line (meta &form))]
     `(clojure.core/let [component# (create-test-component
                                      ~file-name
                                      ~namespace-name
                                      ~definition-line-number
                                      ~@arguments)]
        (platform/register-component component#
                                     ~namespace-name
                                     {::platform/component-version 1}))))))
