(ns coconut.v1.matchers-test
  (:require
    [coconut.v1.core :as c]
    [coconut.v1.matchers :as m]
    [coconut.v1.platform :as platform]
    ))

(defn run-matcher
  [matcher value] (matcher value))

(defn passes-for
  [actual]
  (fn [matcher]
    (when-let [error (m/evaluate-matcher matcher actual)]
      {:expected ["the given matcher to return no error"]
       :actual [(pr-str error)]})))

(defn fails-for
  [actual]
  (fn [matcher]
    (when-not (m/evaluate-matcher matcher actual)
      {:expected ["the given matcher to return an error"]
       :actual ["no error"]})))

(defn has-value
  [value expected]
  (fn [error]
    (when-not (= expected (value error))
      {:expected [(platform/format "the matcher to have an %s value of %s"
                                   (pr-str value)
                                   (pr-str expected))]
       :actual [(pr-str (value error))]})))

(c/describe `c/is
  (c/it "returns no error when the actual is equal to the expected value"
    (fn [assert-that]
      (assert-that (c/is 0) (passes-for 0))))

  (c/it "returns an error when the actual value is not equal to the expected value"
    (fn [assert-that]
      (assert-that (c/is 0) (fails-for 42))))

  (c/it "returns rendered failure values"
    (fn [assert-that]
      (assert-that (run-matcher (c/is 0) 42)
                   (c/contains {:expected ["0"]
                                :actual ["42"]})))))

(c/describe `c/is-not
  (c/it "returns no error when the actual is different from the expected value"
    (fn [assert-that]
      (assert-that (c/is-not 0) (passes-for 1))))

  (c/it "returns an error when the actual value is equal to the expected value"
    (fn [assert-that]
      (assert-that (c/is-not 0) (fails-for 0))))

  (c/it "returns rendered failure values"
    (fn [assert-that]
      (assert-that (run-matcher (c/is-not 0) 0)
                   (c/contains {:expected ["something other than 0"]})))))

(c/describe `c/contains-value
  (c/it "returns no error when the given collection contains the value"
    (fn [assert-that]
      (assert-that (c/contains-value 42) (passes-for (vector 42)))
      (assert-that (c/contains-value 42) (passes-for (hash-set 42)))
      (assert-that (c/contains-value 42) (passes-for (list 42)))))

  (c/it "returns an error when the given collection does not contain the value"
    (fn [assert-that]
      (assert-that (c/contains-value 42) (fails-for (vector 0)))
      (assert-that (c/contains-value 42) (fails-for (hash-set 0)))
      (assert-that (c/contains-value 42) (fails-for (list 0)))))

  (c/it "returns rendered failure values"
    (fn [assert-that]
      (assert-that (run-matcher (c/contains-value 42) [])
                   (has-value :expected ["a collection containing 42"]))

      (assert-that (run-matcher (c/contains-value 42) [])
                   (has-value :actual ["[]"]))

      (assert-that (run-matcher (c/contains-value "42") ["ouch"])
                   (has-value :expected ["a collection containing \"42\""]))

      (assert-that (run-matcher (c/contains-value "42") ["ouch"])
                   (has-value :actual ["[\"ouch\"]"])))))

(c/describe `c/contains
  (c/it "returns no error when the given collection is empty and there are no expected elements"
    (fn [assert-that]
      (assert-that (c/contains []) (passes-for []))))

  (c/it "returns no error when the given collection is not empty and there are no expected elements"
    (fn [assert-that]
      (assert-that (c/contains []) (passes-for [1 2 3]))))

  (c/it "returns no error when the given collection contains the expected element"
    (fn [assert-that]
      (assert-that (c/contains [1]) (passes-for [1]))))

  (c/it "returns an error when the given collection does not contain the expected element"
    (fn [assert-that]
      (assert-that (c/contains [1]) (fails-for []))))

  (c/it "returns an error when the given collection contains all expected elements"
    (fn [assert-that]
      (assert-that (c/contains [1 2 3]) (passes-for [1 2 3]))))

  (c/it "returns an error when the given collection contains some, but not all, of the expected elements"
    (fn [assert-that]
      (assert-that (c/contains [1 2 3]) (fails-for [1 2]))))

  (c/it "returns no error when the actual collection contains any extra items"
    (fn [assert-that]
      (assert-that (c/contains [1 2 3]) (passes-for [1 2 3 4 5]))))

  (c/it "returns rendered failure values"
    (fn [assert-that]
      (assert-that (run-matcher (c/contains [1]) [])
                   (has-value :expected ["a collection containing [1]"]))

      (assert-that (run-matcher (c/contains [1]) [])
                   (has-value :actual ["[]"]))

      (assert-that (run-matcher (c/contains [1 2]) [1])
                   (has-value :expected ["a collection containing [1 2]"]))

      (assert-that (run-matcher (c/contains [1 2 3 4 5]) [])
                   (has-value :expected ["a collection containing [1 2 3 4 5]"])))))

(c/describe `c/satisfies
  (c/it "returns no error when the value satisfies the given predicate"
    (fn [assert-that]
      (assert-that (c/satisfies number?)
                   (passes-for 42))))

  (c/it "returns an error when the value does not satisfy the predicate"
    (fn [assert-that]
      (assert-that (c/satisfies number?)
                   (fails-for ""))))

  (c/it "returns rendered failure values"
    (fn [assert-that]
      (assert-that (run-matcher (c/satisfies number?) "not a number")
                   (has-value :expected ["a value which satisfies number?"]))

      (assert-that (run-matcher (c/satisfies number?) "not a number")
                   (has-value :actual ["\"not a number\""]))

      (assert-that (run-matcher (c/satisfies (partial = 42)) 0)
                   (has-value :expected ["a value which satisfies (partial = 42)"]))

      (assert-that (run-matcher (c/satisfies (partial = 42)) 0)
                   (has-value :actual ["0"])))))

(c/describe `c/throws
  (c/context "when invoked with no arguments"
    (c/it "returns no error when the given function throws any exception"
      (fn [assert-that]
        (assert-that (c/throws)
                     (passes-for #(throw (platform/exception "ouch"))))))

    (c/it "returns an error when the given function does not throw an exception"
      (fn [assert-that]
        (assert-that (c/throws)
                     (fails-for (constantly 42)))))

    #?(:clj
       (c/it "returns rendered failure values"
         (fn [assert-that]
           (assert-that (run-matcher (c/throws) (constantly 42))
                        (c/contains {:expected ["a java.lang.Throwable to be thrown"]
                                     :actual ["no exception"]})))))

    #?(:cljs
       (c/it "returns rendered failure values"
         (fn [assert-that]
           (assert-that (run-matcher (c/throws) (constantly 42))
                        (c/contains {:expected "a Object to be thrown"
                                     :actual "no exception"}))))))

  #?(:clj
      (c/context "when invoked with an exception class"
        (c/it "returns no error when the given function throws the expected exception type"
          (fn [assert-that]
            (assert-that (c/throws NullPointerException)
                         (passes-for #(throw (NullPointerException. "ouch"))))))

        (c/it "returns an error when the given function does not throw an exception"
          (fn [assert-that]
            (assert-that (c/throws NullPointerException)
                         (fails-for (constantly 42)))))

        (c/it "returns an error when the given function throws an exception of a different type"
          (fn [assert-that]
            (assert-that (c/throws NullPointerException)
                         (fails-for #(throw (Exception. "ouch"))))))

        (c/it "returns rendered failure values when no exception is thrown"
          (fn [assert-that]
            (assert-that (run-matcher (c/throws NullPointerException) (constantly 42))
                         (c/contains {:expected ["a java.lang.NullPointerException to be thrown"]
                                      :actual ["no exception"]}))))

        (c/it "returns rendered failure values when the wrong exception type is thrown"
          (fn [assert-that]
            (let [e (Exception. "ouch")]
              (assert-that (run-matcher (c/throws NullPointerException) #(throw e))
                           (c/contains {:expected ["a java.lang.NullPointerException to be thrown"]
                                        :actual (m/render-data e)})))))))

  #?(:cljs
      (c/context "when invoked with an exception class"
        (c/it "returns no error when the given function throws the expected exception type"
          (fn [assert-that]
            (assert-that (c/throws js/TypeError)
                         (passes-for #(throw (js/TypeError. "ouch"))))))

        (c/it "returns an error when the given function does not throw an exception"
          (fn [assert-that]
            (assert-that (c/throws js/TypeError)
                         (fails-for (constantly 42)))))

        (c/it "returns an error when the given function throws an exception of a different type"
          (fn [assert-that]
            (assert-that (c/throws js/TypeError)
                         (fails-for #(throw (js/Error. "ouch"))))))

        (c/it "returns rendered failure values when no exception is thrown"
          (fn [assert-that]
            (assert-that (run-matcher (c/throws js/TypeError) (constantly 42))
                         (c/contains {:expected "a TypeError to be thrown"
                                      :actual "no exception"}))))

        (c/it "returns rendered failure values when the wrong exception type is thrown"
          (fn [assert-that]
            (assert-that (run-matcher (c/throws js/TypeError) #(throw (js/Error. "ouch")))
                         (c/contains {:expected "a TypeError to be thrown"
                                      :actual "Error"}))))))

  #?(:clj
      (c/context "when invoked with an exception class and an expected message"
        (c/it "returns no error when the given function throws the expected exception type"
          (fn [assert-that]
            (assert-that (c/throws NullPointerException "ouch")
                         (passes-for #(throw (NullPointerException. "ouch"))))))

        (c/it "returns an error when the given function does not throw an exception"
          (fn [assert-that]
            (assert-that (c/throws NullPointerException "ouch")
                         (fails-for (constantly 42)))))

        (c/it "returns an error when the given function throws an exception of a different type"
          (fn [assert-that]
            (assert-that (c/throws NullPointerException "ouch")
                         (fails-for #(throw (Exception. "ouch"))))))

        (c/it "returns an error when the given function throws the correct exception with an incorrect message"
          (fn [assert-that]
            (assert-that (c/throws NullPointerException "woot")
                         (fails-for #(throw (NullPointerException. "ouch"))))))

        (c/it "returns rendered failure values when no exception is thrown"
          (fn [assert-that]
            (assert-that (run-matcher (c/throws NullPointerException "woot") (constantly 42))
                         (c/contains {:expected ["a java.lang.NullPointerException to be thrown"]
                                      :actual ["no exception"]}))))

        (c/it "returns rendered failure values when the wrong exception type is thrown"
          (fn [assert-that]
            (let [e (Exception. "ouch")]
              (assert-that (run-matcher (c/throws NullPointerException "woot")
                                        #(throw e))
                           (c/contains {:expected ["a java.lang.NullPointerException to be thrown"]
                                        :actual (m/render-data e)})))))

        (c/it "returns rendered failure values when the exception has the wrong message"
          (fn [assert-that]
            (let [e (NullPointerException. "ouch")]
              (assert-that (run-matcher (c/throws NullPointerException "woot") #(throw e))
                           (c/contains {:expected ["an exception message of \"woot\""]
                                        :actual (m/render-data e)})))))))

  #?(:cljs
      (c/context "when invoked with an exception class and an expected message"
        (c/it "returns no error when the given function throws the expected exception type"
          (fn [assert-that]
            (assert-that (c/throws js/TypeError "ouch")
                         (passes-for #(throw (js/TypeError. "ouch"))))))

        (c/it "returns an error when the given function does not throw an exception"
          (fn [assert-that]
            (assert-that (c/throws js/TypeError "ouch")
                         (fails-for (constantly 42)))))

        (c/it "returns an error when the given function throws an exception of a different type"
          (fn [assert-that]
            (assert-that (c/throws js/TypeError "ouch")
                         (fails-for #(throw (js/Error. "ouch"))))))

        (c/it "returns an error when the given function throws the correct exception with an incorrect message"
          (fn [assert-that]
            (assert-that (c/throws js/TypeError "woot")
                         (fails-for #(throw (js/TypeError. "ouch"))))))

        (c/it "returns rendered failure values when no exception is thrown"
          (fn [assert-that]
            (assert-that (run-matcher (c/throws js/TypeError "woot") (constantly 42))
                         (c/contains {:expected "a TypeError to be thrown"
                                      :actual "no exception"}))))

        (c/it "returns rendered failure values when the wrong exception type is thrown"
          (fn [assert-that]
            (assert-that (run-matcher (c/throws js/TypeError "woot") #(throw (js/Error. "ouch")))
                         (c/contains {:expected "a TypeError to be thrown"
                                      :actual "Error"}))))

        (c/it "returns rendered failure values when the exception has the wrong message"
          (fn [assert-that]
            (assert-that (run-matcher (c/throws js/TypeError "woot") #(throw (js/TypeError. "ouch")))
                         (c/contains {:expected "an exception message of \"woot\""
                                      :actual "\"ouch\""})))))))
