(ns coconut.v1.progress
  #?(:cljs (:require-macros [cljs.core.async.macros :as async]))
  (:require
    [coconut.v1.platform :as platform]
    [coconut.v1.summarizing :as summarizing]
    [coconut.v1.rendering :as rendering]
    [coconut.v1.util :as util]
    [clojure.core.async :as async :refer [<! >!]]
    [clojure.pprint :as cp]
    [clojure.string :as cs]
    ))

(defn ^{:private true} with-indicies
  ([elements]
   (into (vector)
         (comp (map-indexed vector)
               (map #(update % 0 inc)))
         elements)))

(def ^{:private true :const true} checkmark \✔)
(def ^{:private true :const true} x \✘)
(def ^{:private true :const true} asterisk \*)

(defmulti ^{:private true} report-event
  (fn [state event output-channel]
    (::summarizing/type event)))

(defmethod ^{:private true} report-event
  ::summarizing/suite-started
  ([state event output-channel]
   (async/go
     (merge state
            #::{:total-number-of-tests (::summarizing/total-number-of-tests event)}))))

(defmethod ^{:private true} report-event
  ::summarizing/context-started
  ([state event output-channel]
   (async/go
     (update state
             ::level inc))))

(defmethod ^{:private true} report-event
  ::summarizing/context-finished
  ([state event output-channel]
   (async/go
     (update state
             ::level dec))))

(defmethod ^{:private true} report-event
  ::summarizing/test-pending
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/segment
                      :segments [#::rendering{:text asterisk
                                              :color ::rendering/yellow}]})
     (update state
             ::pending-tests conj event))))

(defmethod ^{:private true} report-event
  ::summarizing/test-passed
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/segment
                      :segments [#::rendering{:text \.
                                              :color ::rendering/green}]})
     state)))

(defmethod ^{:private true} report-event
  ::summarizing/test-failed
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/segment
                      :segments [#::rendering{:text \F
                                              :color ::rendering/red}]})
     (update state
             ::failing-tests conj event))))

(defmethod ^{:private true} report-event
  ::summarizing/test-threw-exception
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/segment
                      :segments [#::rendering{:text \F
                                              :color ::rendering/red}]})
     (update state
             ::failing-tests conj event))))

(defmethod ^{:private true} report-event
  ::summarizing/test-timed-out
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/segment
                      :segments [#::rendering{:text \F
                                              :color ::rendering/red}]})
     (update state
             ::failing-tests conj event))))

(defmethod ^{:private true} report-event
  ::summarizing/suite-finished
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/newline}))))

(defmethod ^{:private true} report-event
  :default
  ([state event output-channel]
   (async/go state)))

(defn report
  "Given a core.async channel of summarized test events, returns a new channel
  containing data structures describing the information which should be printed
  to the screen."
  ([event-channel]
   (util/run-consumers event-channel [report]))
  ([event-channel output-channel]
   (async/go
     (loop [state #::{:total-number-of-tests nil
                      :pending-tests []
                      :failing-tests []
                      :level 0}]
       (when-let [event (<! event-channel)]
         (let [updated-state (<! (report-event state event output-channel))]
           (recur updated-state)))))))
