(ns coconut.v1.test-namespaces.fixtures
  (:require
    [coconut.v1.core :as c]
    [coconut.v1.running :as r]
    [coconut.v1.summarizing :as s]
    [coconut.v1.platform :as platform]
    ))

(c/deftest "a synchronous test with no assertions"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test with no assertions"}
                   #::r{:type ::r/test-finished
                        :description "a synchronous test with no assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-pending
                        :context []
                        :description "a synchronous test with no assertions"
                        :pending-reason "nothing has been asserted"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]))

(c/deftest "an asynchronous test with no assertions"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test with no assertions"}
                   #::r{:type ::r/test-finished
                        :description "an asynchronous test with no assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-pending
                        :context []
                        :description "an asynchronous test with no assertions"
                        :pending-reason "nothing has been asserted"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate done)))

(c/deftest "a synchronous test which is marked pending"
  {:pending "for some reason"
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-marked-pending
                        :description "a synchronous test which is marked pending"
                        :pending-reason "for some reason"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-pending
                        :context []
                        :description "a synchronous test which is marked pending"
                        :pending-reason "for some reason"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]))

(c/deftest "an asynchronous test which is marked pending and invokes the done function"
  {:pending "for some reason"
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-marked-pending
                        :description "an asynchronous test which is marked pending and invokes the done function"
                        :pending-reason "for some reason"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-pending
                        :context []
                        :description "an asynchronous test which is marked pending and invokes the done function"
                        :pending-reason "for some reason"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate done)))

(c/deftest "an asynchronous test which is marked pending but does not invoke the done function"
  {:pending "for some reason"
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-marked-pending
                        :description "an asynchronous test which is marked pending but does not invoke the done function"
                        :pending-reason "for some reason"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-pending
                        :context []
                        :description "an asynchronous test which is marked pending but does not invoke the done function"
                        :pending-reason "for some reason"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]))

(c/deftest "a synchronous test with one passing assertion"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test with one passing assertion"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "a synchronous test with one passing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "a synchronous test with one passing assertion"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]
    (assert-that 42 (c/is 42))))

(c/deftest "an asynchronous test with one passing assertion"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test with one passing assertion"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "an asynchronous test with one passing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test with one passing assertion"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(do (assert-that 42 (c/is 42))
           (done)))))

(c/deftest "a synchronous test with multiple passing assertions"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test with multiple passing assertions"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "a synchronous test with multiple passing assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "a synchronous test with multiple passing assertions"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]
    (assert-that 42 (c/is 42))
    (assert-that 42 (c/is 42))))

(c/deftest "an asynchronous test with multiple passing assertions"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test with multiple passing assertions"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "an asynchronous test with multiple passing assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test with multiple passing assertions"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(do (assert-that 42 (c/is 42))
           (assert-that 42 (c/is 42))
           (done)))))

(c/deftest "a synchronous test with one failing assertion"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test with one failing assertion"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-finished
                        :description "a synchronous test with one failing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "a synchronous test with one failing assertion"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]
    (assert-that 0 (c/is 42))))

(c/deftest "an asynchronous test with one failing assertion"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test with one failing assertion"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-finished
                        :description "an asynchronous test with one failing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test with one failing assertion"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(do (assert-that 0 (c/is 42))
           (done)))))

(c/deftest "a synchronous test with multiple failing assertions"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test with multiple failing assertions"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-finished
                        :description "a synchronous test with multiple failing assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "a synchronous test with multiple failing assertions"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]
    (assert-that 0 (c/is 42))
    (assert-that 0 (c/is 42))))

(c/deftest "an asynchronous test with multiple failing assertions"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test with multiple failing assertions"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-finished
                        :description "an asynchronous test with multiple failing assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test with multiple failing assertions"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(do (assert-that 0 (c/is 42))
           (assert-that 0 (c/is 42))
           (done)))))

(c/deftest "a synchronous test which passes then fails an assertion"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test which passes then fails an assertion"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-finished
                        :description "a synchronous test which passes then fails an assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "a synchronous test which passes then fails an assertion"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]
    (assert-that 42 (c/is 42))
    (assert-that 0 (c/is 42))))

(c/deftest "an asynchronous test which passes then fails an assertion"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which passes then fails an assertion"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-finished
                        :description "an asynchronous test which passes then fails an assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test which passes then fails an assertion"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(do (assert-that 42 (c/is 42))
           (assert-that 0 (c/is 42))
           (done)))))

(c/deftest "a synchronous test which fails then passes an assertion"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test which fails then passes an assertion"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "a synchronous test which fails then passes an assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "a synchronous test which fails then passes an assertion"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]
    (assert-that 0 (c/is 42))
    (assert-that 42 (c/is 42))))

(c/deftest "an asynchronous test which fails then passes an assertion"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which fails then passes an assertion"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "an asynchronous test which fails then passes an assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test which fails then passes an assertion"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(do (assert-that 0 (c/is 42))
           (assert-that 42 (c/is 42))
           (done)))))

(c/deftest "a synchronous test which throws an exception"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test which throws an exception"}
                   #::r{:type ::r/test-threw-exception
                        :description "a synchronous test which throws an exception"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-threw-exception
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "a synchronous test which throws an exception"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]
    (throw (platform/exception "ouch"))))

(c/deftest "an asynchronous test which synchronously throws an exception"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which synchronously throws an exception"}
                   #::r{:type ::r/test-threw-exception
                        :description "an asynchronous test which synchronously throws an exception"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-threw-exception
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test which synchronously throws an exception"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (throw (platform/exception "ouch"))))

(c/deftest "an asynchronous test which invokes the done function with an exception"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which invokes the done function with an exception"}
                   #::r{:type ::r/test-threw-exception
                        :description "an asynchronous test which invokes the done function with an exception"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-threw-exception
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test which invokes the done function with an exception"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(try (do (throw (platform/exception "ouch"))
                (done))
            (catch #?(:clj Throwable :cljs :default) e
              (done e))))))

; (c/deftest "an asynchronous test which invokes the done function with an exception"
;   {:asynchronous {:timeout 50}
;    :tags #{:fixture}
;    ::r/event-data [#::r{:type ::r/suite-started}
;                         #::r{:type ::r/test-started
;                              :namespace-name "coconut.v1.test-namespaces.fixtures"
;                              :description "an asynchronous test which invokes the done function with an exception"}
;                         #::r{:type ::r/test-threw-exception
;                              :description "an asynchronous test which invokes the done function with an exception"}
;                         #::r{:type ::r/suite-finished}]
;    ::s/event-data [#::s{:type ::s/suite-started}
;                         #::s{:type ::s/test-threw-exception
;                              :namespace-name "coconut.v1.test-namespaces.fixtures"
;                              :context []
;                              :description "an asynchronous test which invokes the done function with an exception"}
;                         #::s{:type ::s/suite-finished}]}
;   (fn [assert-that done]
;     (platform/set-immediate
;       #(do (throw (platform/exception "ouch"))
;            (done)))))

(c/deftest "a synchronous test which throws an exception after a passing assertion"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test which throws an exception after a passing assertion"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-threw-exception
                        :description "a synchronous test which throws an exception after a passing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-threw-exception
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "a synchronous test which throws an exception after a passing assertion"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]
    (assert-that 42 (c/is 42))
    (throw (platform/exception "ouch"))))

(c/deftest "an asynchronous test which synchronously throws an exception after a passing assertion"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which synchronously throws an exception after a passing assertion"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-threw-exception
                        :description "an asynchronous test which synchronously throws an exception after a passing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-threw-exception
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test which synchronously throws an exception after a passing assertion"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (assert-that 42 (c/is 42))
    (throw (platform/exception "ouch"))
    (done)))

(c/deftest "an asynchronous test which invokes the done function with an exception after a passing assertion"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which invokes the done function with an exception after a passing assertion"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-threw-exception
                        :description "an asynchronous test which invokes the done function with an exception after a passing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-threw-exception
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test which invokes the done function with an exception after a passing assertion"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(try (do (assert-that 42 (c/is 42))
                (throw (platform/exception "ouch"))
                (done))
            (catch #?(:clj Throwable :cljs :default) e
              (done e))))))

; (c/deftest "an asynchronous test which throws an exception after a passing assertion"
;   {:asynchronous {:timeout 50}
;    :tags #{:fixture}
;    ::r/event-data [#::r{:type ::r/suite-started}
;                         #::r{:type ::r/test-started
;                              :namespace-name "coconut.v1.test-namespaces.fixtures"
;                              :description "an asynchronous test which throws an exception after a passing assertion"}
;                         #::r{:type ::r/assertion-passed}
;                         #::r{:type ::r/test-threw-exception
;                              :description "an asynchronous test which throws an exception after a passing assertion"}
;                         #::r{:type ::r/suite-finished}]
;    ::s/event-data [#::s{:type ::s/suite-started}
;                         #::s{:type ::s/test-threw-exception
;                              :namespace-name "coconut.v1.test-namespaces.fixtures"
;                              :context []
;                              :description "an asynchronous test which throws an exception after a passing assertion"}
;                         #::s{:type ::s/suite-finished}]}
;   (fn [assert-that done]
;     (platform/set-immediate
;       #(do (assert-that 42 (c/is 42))
;            (throw (platform/exception "ouch"))
;            (done)))))

(c/deftest "a synchronous test which throws an exception after a failing assertion"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test which throws an exception after a failing assertion"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-threw-exception
                        :description "a synchronous test which throws an exception after a failing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "a synchronous test which throws an exception after a failing assertion"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]
    (assert-that 0 (c/is 42))
    (throw (platform/exception "ouch"))))

(c/deftest "an asynchronous test which synchronously throws an exception after a failing assertion"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which synchronously throws an exception after a failing assertion"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-threw-exception
                        :description "an asynchronous test which synchronously throws an exception after a failing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test which synchronously throws an exception after a failing assertion"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (assert-that 0 (c/is 42))
    (throw (platform/exception "ouch"))
    (done)))

(c/deftest "an asynchronous test which invokes the done function with an exception after a failing assertion"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which invokes the done function with an exception after a failing assertion"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-threw-exception
                        :description "an asynchronous test which invokes the done function with an exception after a failing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test which invokes the done function with an exception after a failing assertion"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(try (do (assert-that 0 (c/is 42))
                (throw (platform/exception "ouch"))
                (done))
            (catch #?(:clj Throwable :cljs :default) e
              (done e))))))

(c/deftest "a synchronous test which throws an exception after various passing/failing assertions"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "a synchronous test which throws an exception after various passing/failing assertions"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-threw-exception
                        :description "a synchronous test which throws an exception after various passing/failing assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "a synchronous test which throws an exception after various passing/failing assertions"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that]
    (assert-that 42 (c/is 42))
    (assert-that 0 (c/is 42))
    (assert-that 42 (c/is 42))
    (throw (platform/exception "ouch"))))

(c/deftest "an asynchronous test which synchronously throws an exception after various passing/failing assertions"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which synchronously throws an exception after various passing/failing assertions"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-threw-exception
                        :description "an asynchronous test which synchronously throws an exception after various passing/failing assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test which synchronously throws an exception after various passing/failing assertions"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (assert-that 42 (c/is 42))
    (assert-that 0 (c/is 42))
    (assert-that 42 (c/is 42))
    (throw (platform/exception "ouch"))
    (done)))

(c/deftest "an asynchronous test which invokes the done function with an exception after various passing/failing assertions"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which invokes the done function with an exception after various passing/failing assertions"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-threw-exception
                        :description "an asynchronous test which invokes the done function with an exception after various passing/failing assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context []
                        :description "an asynchronous test which invokes the done function with an exception after various passing/failing assertions"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(try (do (assert-that 42 (c/is 42))
                (assert-that 0 (c/is 42))
                (assert-that 42 (c/is 42))
                (throw (platform/exception "ouch"))
                (done))
            (catch #?(:clj Throwable :cljs :default) e
              (done e))))))

; (c/deftest "an asynchronous test which throws an exception after various passing/failing assertions"
;   {:asynchronous {:timeout 50}
;    :tags #{:fixture}
;    ::r/event-data [#::r{:type ::r/suite-started}
;                         #::r{:type ::r/test-started
;                              :namespace-name "coconut.v1.test-namespaces.fixtures"
;                              :description "an asynchronous test which throws an exception after various passing/failing assertions"}
;                         #::r{:type ::r/assertion-passed}
;                         #::r{:type ::r/assertion-failed
;                              :expected "42"
;                              :actual "0"}
;                         #::r{:type ::r/test-threw-exception
;                              :description "an asynchronous test which throws an exception after various passing/failing assertions"}
;                         #::r{:type ::r/suite-finished}]
;    ::s/event-data [#::s{:type ::s/suite-started}
;                         #::s{:type ::s/test-threw-exception
;                              :namespace-name "coconut.v1.test-namespaces.fixtures"
;                              :context []
;                              :description "an asynchronous test which throws an exception after various passing/failing assertions"}
;                         #::s{:type ::s/suite-finished}]}
;   (fn [assert-that done]
;     (platform/set-immediate
;       #(do (assert-that 42 (c/is 42))
;            (assert-that 0 (c/is 42))
;            (throw (platform/exception "ouch"))
;            (done)))))

(c/deftest "an asynchronous test which times out after no assertions"
  {:asynchronous {:timeout 1}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which times out after no assertions"}
                   #::r{:type ::r/test-timed-out
                        :description "an asynchronous test which times out after no assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-timed-out
                        :context []
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which times out after no assertions"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]))

(c/deftest "an asynchronous test which times out after a single passing assertion"
  {:asynchronous {:timeout 5}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which times out after a single passing assertion"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-timed-out
                        :description "an asynchronous test which times out after a single passing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-timed-out
                        :context []
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which times out after a single passing assertion"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(assert-that 42 (c/is 42)))))

(c/deftest "an asynchronous test which times out after multiple passing assertions"
  {:asynchronous {:timeout 5}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which times out after multiple passing assertions"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-timed-out
                        :description "an asynchronous test which times out after multiple passing assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-timed-out
                        :context []
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which times out after multiple passing assertions"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(do (assert-that 42 (c/is 42))
           (assert-that 42 (c/is 42))))))

(c/deftest "an asynchronous test which times out after a single failing assertion"
  {:asynchronous {:timeout 5}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which times out after a single failing assertion"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-timed-out
                        :description "an asynchronous test which times out after a single failing assertion"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-timed-out
                        :context []
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which times out after a single failing assertion"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(assert-that 0 (c/is 42)))))

(c/deftest "an asynchronous test which times out after multiple passing assertions"
  {:asynchronous {:timeout 5}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which times out after multiple passing assertions"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-timed-out
                        :description "an asynchronous test which times out after multiple passing assertions"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/test-timed-out
                        :context []
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "an asynchronous test which times out after multiple passing assertions"}
                   #::s{:type ::s/suite-finished}]}
  (fn [assert-that done]
    (platform/set-immediate
      #(do (assert-that 0 (c/is 42))
           (assert-that 0 (c/is 42))))))

(c/describe "an empty context"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/suite-finished}]})

(c/describe "nested empty contexts"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/suite-finished}]}
  (c/context "one")
  (c/context "two"))

(c/describe "a context containing tests with various outcomes"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a context containing tests with various outcomes"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is pending"}
                   #::r{:type ::r/test-finished
                        :description "is pending"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is failing"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-finished
                        :description "is failing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "throws an exception"}
                   #::r{:type ::r/test-threw-exception
                        :description "throws an exception"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "times out"}
                   #::r{:type ::r/test-timed-out
                        :description "times out"}
                   #::r{:type ::r/context-finished
                        :subject "a context containing tests with various outcomes"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a context containing tests with various outcomes"}
                   #::s{:type ::s/test-pending
                        :context ["a context containing tests with various outcomes"]
                        :description "is pending"
                        :pending-reason "nothing has been asserted"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context containing tests with various outcomes"]
                        :description "is passing"}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context containing tests with various outcomes"]
                        :description "is failing"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/test-threw-exception
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context containing tests with various outcomes"]
                        :description "throws an exception"}
                   #::s{:type ::s/test-timed-out
                        :context ["a context containing tests with various outcomes"]
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "times out"}
                   #::s{:type ::s/context-finished
                        :subject "a context containing tests with various outcomes"}
                   #::s{:type ::s/suite-finished}]}
  (c/it "is pending"
    (fn [assert-that]))

  (c/it "is passing"
    (fn [assert-that]
      (assert-that 42 (c/is 42))))

  (c/it "is failing"
    (fn [assert-that]
      (assert-that 0 (c/is 42))))

  (c/it "throws an exception"
    (fn [assert-that]
      (throw (platform/exception "ouch"))))

  (c/it "times out"
    {:asynchronous {:timeout 1}}
    (fn [assert-that done])))

(c/describe "a nested context containing tests with various outcomes"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a nested context containing tests with various outcomes"}
                   #::r{:type ::r/context-started
                        :subject "a nested context"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is pending"}
                   #::r{:type ::r/test-finished
                        :description "is pending"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is failing"}
                   #::r{:type ::r/assertion-failed
                        :expected "42"
                        :actual "0"}
                   #::r{:type ::r/test-finished
                        :description "is failing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "throws an exception"}
                   #::r{:type ::r/test-threw-exception
                        :description "throws an exception"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "times out"}
                   #::r{:type ::r/test-timed-out
                        :description "times out"}
                   #::r{:type ::r/context-finished
                        :subject "a nested context"}
                   #::r{:type ::r/context-finished
                        :subject "a nested context containing tests with various outcomes"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a nested context containing tests with various outcomes"}
                   #::s{:type ::s/context-started
                        :subject "a nested context"}
                   #::s{:type ::s/test-pending
                        :context ["a nested context containing tests with various outcomes"
                                  "a nested context"]
                        :description "is pending"
                        :pending-reason "nothing has been asserted"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a nested context containing tests with various outcomes"
                                  "a nested context"]
                        :description "is passing"}
                   #::s{:type ::s/test-failed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a nested context containing tests with various outcomes"
                                  "a nested context"]
                        :description "is failing"
                        :expected "42"
                        :actual "0"}
                   #::s{:type ::s/test-threw-exception
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a nested context containing tests with various outcomes"
                                  "a nested context"]
                        :description "throws an exception"}
                   #::s{:type ::s/test-timed-out
                        :context ["a nested context containing tests with various outcomes"
                                  "a nested context"]}
                   #::s{:type ::s/context-finished
                        :subject "a nested context"}
                   #::s{:type ::s/context-finished
                        :subject "a nested context containing tests with various outcomes"}
                   #::s{:type ::s/suite-finished}]}
  (c/context "a nested context"
    (c/it "is pending"
      (fn [assert-that]))

    (c/it "is passing"
      (fn [assert-that]
        (assert-that 42 (c/is 42))))

    (c/it "is failing"
      (fn [assert-that]
        (assert-that 0 (c/is 42))))

    (c/it "throws an exception"
      (fn [assert-that]
        (throw (platform/exception "ouch"))))

    (c/it "times out"
      {:asynchronous {:timeout 1}}
      (fn [assert-that done]))))

(c/describe "an asynchronous context"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "an asynchronous context"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is asynchronous"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is asynchronous"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is asynchronous"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is asynchronous"}
                   #::r{:type ::r/context-finished
                        :subject "an asynchronous context"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "an asynchronous context"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["an asynchronous context"]
                        :description "is asynchronous"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["an asynchronous context"]
                        :description "is asynchronous"}
                   #::s{:type ::s/context-finished
                        :subject "an asynchronous context"}
                   #::s{:type ::s/suite-finished}]}
  (c/it "is asynchronous"
    (fn [assert-that done]
      (platform/set-immediate
        #(do (assert-that 42 (c/is 42))
             (done)))))

  (c/it "is asynchronous"
    (fn [assert-that done]
      (platform/set-immediate
        #(do (assert-that 42 (c/is 42))
             (done))))))

(c/describe "an asynchronous context with a nested context"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "an asynchronous context with a nested context"}
                   #::r{:type ::r/context-started
                        :subject "does not specify if it is asynchronous or not"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is asynchronous"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is asynchronous"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is asynchronous"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is asynchronous"}
                   #::r{:type ::r/context-finished
                        :subject "does not specify if it is asynchronous or not"}
                   #::r{:type ::r/context-finished
                        :subject "an asynchronous context with a nested context"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "an asynchronous context with a nested context"}
                   #::s{:type ::s/context-started
                        :subject "does not specify if it is asynchronous or not"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["an asynchronous context with a nested context"
                                  "does not specify if it is asynchronous or not"]
                        :description "is asynchronous"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["an asynchronous context with a nested context"
                                  "does not specify if it is asynchronous or not"]
                        :description "is asynchronous"}
                   #::s{:type ::s/context-finished
                        :subject "does not specify if it is asynchronous or not"}
                   #::s{:type ::s/context-finished
                        :subject "an asynchronous context with a nested context"}
                   #::s{:type ::s/suite-finished}]}
  (c/context "does not specify if it is asynchronous or not"
    (c/it "is asynchronous"
      (fn [assert-that done]
        (platform/set-immediate
          #(do (assert-that 42 (c/is 42))
               (done)))))

    (c/it "is asynchronous"
      (fn [assert-that done]
        (platform/set-immediate
          #(do (assert-that 42 (c/is 42))
               (done)))))))

(c/describe "an asynchronous context with tests that are not asynchronous"
  {:asynchronous {:timeout 50}
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "an asynchronous context with tests that are not asynchronous"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is not asynchronous"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is not asynchronous"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is not asynchronous"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is not asynchronous"}
                   #::r{:type ::r/context-finished
                        :subject "an asynchronous context with tests that are not asynchronous"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "an asynchronous context with tests that are not asynchronous"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["an asynchronous context with tests that are not asynchronous"]
                        :description "is not asynchronous"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["an asynchronous context with tests that are not asynchronous"]
                        :description "is not asynchronous"}
                   #::s{:type ::s/context-finished
                        :subject "an asynchronous context with tests that are not asynchronous"}
                   #::s{:type ::s/suite-finished}]}
  (c/it "is not asynchronous"
    {:asynchronous false}
    (fn [assert-that]
      (assert-that 42 (c/is 42))))

  (c/it "is not asynchronous"
    {:asynchronous false}
    (fn [assert-that]
      (assert-that 42 (c/is 42)))))

(c/describe "a pending context"
  {:pending "for some reason"
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a pending context"}
                   #::r{:type ::r/test-marked-pending
                        :description "is pending"
                        :pending-reason "for some reason"}
                   #::r{:type ::r/test-marked-pending
                        :description "is pending"
                        :pending-reason "for some reason"}
                   #::r{:type ::r/context-finished
                        :subject "a pending context"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a pending context"}
                   #::s{:type ::s/test-pending
                        :context ["a pending context"]
                        :description "is pending"
                        :pending-reason "for some reason"}
                   #::s{:type ::s/test-pending
                        :context ["a pending context"]
                        :description "is pending"
                        :pending-reason "for some reason"}
                   #::s{:type ::s/context-finished
                        :subject "a pending context"}
                   #::s{:type ::s/suite-finished}]}
  (c/it "is pending"
    (fn [assert-that]
      (assert-that 42 (c/is 42))))

  (c/it "is pending"
    (fn [assert-that]
      (assert-that 42 (c/is 42)))))

(c/describe "a pending context with a nested context"
  {:pending "for some reason"
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a pending context with a nested context"}
                   #::r{:type ::r/context-started
                        :subject "does not specify if it is pending or not"}
                   #::r{:type ::r/test-marked-pending
                        :description "is pending"
                        :pending-reason "for some reason"}
                   #::r{:type ::r/test-marked-pending
                        :description "is pending"
                        :pending-reason "for some reason"}
                   #::r{:type ::r/context-finished
                        :subject "does not specify if it is pending or not"}
                   #::r{:type ::r/context-finished
                        :subject "a pending context with a nested context"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a pending context with a nested context"}
                   #::s{:type ::s/context-started
                        :subject "does not specify if it is pending or not"}
                   #::s{:type ::s/test-pending
                        :context ["a pending context with a nested context"
                                  "does not specify if it is pending or not"]
                        :description "is pending"
                        :pending-reason "for some reason"}
                   #::s{:type ::s/test-pending
                        :context ["a pending context with a nested context"
                                  "does not specify if it is pending or not"]
                        :description "is pending"
                        :pending-reason "for some reason"}
                   #::s{:type ::s/context-finished
                        :subject "does not specify if it is pending or not"}
                   #::s{:type ::s/context-finished
                        :subject "a pending context with a nested context"}
                   #::s{:type ::s/suite-finished}]}
  (c/context "does not specify if it is pending or not"
    (c/it "is pending"
      (fn [assert-that]
        (assert-that 42 (c/is 42))))

    (c/it "is pending"
      (fn [assert-that]
        (assert-that 42 (c/is 42))))))

(c/describe "a pending context with tests that are not pending"
  {:pending "for some reason"
   :tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a pending context with tests that are not pending"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is not pending"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is not pending"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is not pending"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is not pending"}
                   #::r{:type ::r/context-finished
                        :subject "a pending context with tests that are not pending"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a pending context with tests that are not pending"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a pending context with tests that are not pending"]
                        :description "is not pending"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a pending context with tests that are not pending"]
                        :description "is not pending"}
                   #::s{:type ::s/context-finished
                        :subject "a pending context with tests that are not pending"}
                   #::s{:type ::s/suite-finished}]}
  (c/it "is not pending"
    {:pending false}
    (fn [assert-that]
      (assert-that 42 (c/is 42))))

  (c/it "is not pending"
    {:pending false}
    (fn [assert-that]
      (assert-that 42 (c/is 42)))))

(c/describe "a context with tests defined in a let"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a context with tests defined in a let"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/context-finished
                        :subject "a context with tests defined in a let"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a context with tests defined in a let"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with tests defined in a let"]
                        :description "is passing"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with tests defined in a let"]
                        :description "is passing"}
                   #::s{:type ::s/context-finished
                        :subject "a context with tests defined in a let"}
                   #::s{:type ::s/suite-finished}]}
  (c/let [x 42]
    (c/it "is passing"
      (fn [assert-that]
        (assert-that x (c/is 42))))

    (c/it "is passing"
      (fn [assert-that]
        (assert-that x (c/is 42))))))

(c/describe "a context with a sub-context defined in a let"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a context with a sub-context defined in a let"}
                   #::r{:type ::r/context-started
                        :subject "sub context"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/context-finished
                        :subject "sub context"}
                   #::r{:type ::r/context-finished
                        :subject "a context with a sub-context defined in a let"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a context with a sub-context defined in a let"}
                   #::s{:type ::s/context-started
                        :subject "sub context"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with a sub-context defined in a let"
                                  "sub context"]
                        :description "is passing"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with a sub-context defined in a let"
                                  "sub context"]
                        :description "is passing"}
                   #::s{:type ::s/context-finished
                        :subject "sub context"}
                   #::s{:type ::s/context-finished
                        :subject "a context with a sub-context defined in a let"}
                   #::s{:type ::s/suite-finished}]}
  (c/let [x 42]
    (c/context "sub context"
      (c/it "is passing"
        (fn [assert-that]
          (assert-that x (c/is 42))))

      (c/it "is passing"
        (fn [assert-that]
          (assert-that x (c/is 42)))))))

(c/describe "a context with tests defined in a for"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a context with tests defined in a for"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/context-finished
                        :subject "a context with tests defined in a for"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a context with tests defined in a for"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with tests defined in a for"]
                        :description "is passing"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with tests defined in a for"]
                        :description "is passing"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with tests defined in a for"]
                        :description "is passing"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with tests defined in a for"]
                        :description "is passing"}
                   #::s{:type ::s/context-finished
                        :subject "a context with tests defined in a for"}
                   #::s{:type ::s/suite-finished}]}
  (c/for [x [0 1]
          y [1 1]]
    (c/it "is passing"
      (fn [assert-that]
        (assert-that x (c/is x))
        (assert-that y (c/is y))))))

(c/describe "a context with a sub-context defined in a for"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a context with a sub-context defined in a for"}
                   #::r{:type ::r/context-started
                        :subject "sub context"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/context-finished
                        :subject "sub context"}
                   #::r{:type ::r/context-started
                        :subject "sub context"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/context-finished
                        :subject "sub context"}
                   #::r{:type ::r/context-started
                        :subject "sub context"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/context-finished
                        :subject "sub context"}
                   #::r{:type ::r/context-started
                        :subject "sub context"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/context-finished
                        :subject "sub context"}
                   #::r{:type ::r/context-finished
                        :subject "a context with a sub-context defined in a for"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a context with a sub-context defined in a for"}
                   #::s{:type ::s/context-started
                        :subject "sub context"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with a sub-context defined in a for"
                                  "sub context"]
                        :description "is passing"}
                   #::s{:type ::s/context-finished
                        :subject "sub context"}
                   #::s{:type ::s/context-started
                        :subject "sub context"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with a sub-context defined in a for"
                                  "sub context"]
                        :description "is passing"}
                   #::s{:type ::s/context-finished
                        :subject "sub context"}
                   #::s{:type ::s/context-started
                        :subject "sub context"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with a sub-context defined in a for"
                                  "sub context"]
                        :description "is passing"}
                   #::s{:type ::s/context-finished
                        :subject "sub context"}
                   #::s{:type ::s/context-started
                        :subject "sub context"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with a sub-context defined in a for"
                                  "sub context"]
                        :description "is passing"}
                   #::s{:type ::s/context-finished
                        :subject "sub context"}
                   #::s{:type ::s/context-finished
                        :subject "a context with a sub-context defined in a for"}
                   #::s{:type ::s/suite-finished}]}
  (c/for [x [0 1]
          y [0 1]]
    (c/context "sub context"
      (c/it "is passing"
        (fn [assert-that]
          (assert-that x (c/is x))
          (assert-that y (c/is y)))))))

(c/describe "a context with before/after all components"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a context with before/after all components"}
                   #::r{:type ::r/context-started
                        :subject "depends on global state"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/context-finished
                        :subject "depends on global state"}
                   #::r{:type ::r/context-finished
                        :subject "a context with before/after all components"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a context with before/after all components"}
                   #::s{:type ::s/context-started
                        :subject "depends on global state"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with before/after all components"
                                  "depends on global state"]
                        :description "is passing"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with before/after all components"
                                  "depends on global state"]
                        :description "is passing"}
                   #::s{:type ::s/context-finished
                        :subject "depends on global state"}
                   #::s{:type ::s/context-finished
                        :subject "a context with before/after all components"}
                   #::s{:type ::s/suite-finished}]}
  (c/let [n (atom 0)]
    (c/context "depends on global state"
      (c/before-all
        (fn []
          (swap! n inc)))

      (c/after-all
        (fn []
          (reset! n 0)))

      (c/it "is passing"
        (fn [assert-that]
          (assert-that @n (c/is 1))))

      (c/it "is passing"
        (fn [assert-that]
          (assert-that @n (c/is 1)))))))

(c/describe "a context with a before each component"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a context with a before each component"}
                   #::r{:type ::r/context-started
                        :subject "depends on global state"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/context-finished
                        :subject "depends on global state"}
                   #::r{:type ::r/context-finished
                        :subject "a context with a before each component"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a context with a before each component"}
                   #::s{:type ::s/context-started
                        :subject "depends on global state"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with a before each component"
                                  "depends on global state"]
                        :description "is passing"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with a before each component"
                                  "depends on global state"]
                        :description "is passing"}
                   #::s{:type ::s/context-finished
                        :subject "depends on global state"}
                   #::s{:type ::s/context-finished
                        :subject "a context with a before each component"}
                   #::s{:type ::s/suite-finished}]}
  (c/let [n (atom 0)]
    (c/context "depends on global state"
      (c/before-each
        (fn []
          (swap! n inc)))

      (c/after-all
        (fn []
          (reset! n 0)))

      (c/it "is passing"
        (fn [assert-that]
          (assert-that @n (c/is 1))))

      (c/it "is passing"
        (fn [assert-that]
          (assert-that @n (c/is 2)))))))

(c/describe "a context with an after each component"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a context with an after each component"}
                   #::r{:type ::r/context-started
                        :subject "depends on global state"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is passing"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "is passing"}
                   #::r{:type ::r/context-finished
                        :subject "depends on global state"}
                   #::r{:type ::r/context-finished
                        :subject "a context with an after each component"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a context with an after each component"}
                   #::s{:type ::s/context-started
                        :subject "depends on global state"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with an after each component"
                                  "depends on global state"]
                        :description "is passing"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with an after each component"
                                  "depends on global state"]
                        :description "is passing"}
                   #::s{:type ::s/context-finished
                        :subject "depends on global state"}
                   #::s{:type ::s/context-finished
                        :subject "a context with an after each component"}
                   #::s{:type ::s/suite-finished}]}
  (c/let [n (atom 0)]
    (c/context "depends on global state"
      (c/after-each
        (fn []
          (swap! n inc)))

      (c/after-all
        (fn []
          (reset! n 0)))

      (c/it "is passing"
        (fn [assert-that]
          (assert-that @n (c/is 0))))

      (c/it "is passing"
        (fn [assert-that]
          (assert-that @n (c/is 1)))))))

(c/describe "a context with multiple nested before/after components"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a context with multiple nested before/after components"}
                   #::r{:type ::r/context-started
                        :subject "depends on global state"}
                   #::r{:type ::r/context-started
                        :subject "a"}
                   #::r{:type ::r/context-started
                        :subject "b"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "is a test"}
                   #::r{:type ::r/test-finished
                        :description "is a test"}
                   #::r{:type ::r/context-finished
                        :subject "b"}
                   #::r{:type ::r/context-finished
                        :subject "a"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "runs the before/after components in the specified order"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "runs the before/after components in the specified order"}
                   #::r{:type ::r/context-finished
                        :subject "depends on global state"}
                   #::r{:type ::r/context-finished
                        :subject "a context with multiple nested before/after components"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a context with multiple nested before/after components"}
                   #::s{:type ::s/context-started
                        :subject "depends on global state"}
                   #::s{:type ::s/context-started
                        :subject "a"}
                   #::s{:type ::s/context-started
                        :subject "b"}
                   #::s{:type ::s/test-pending
                        :context ["a context with multiple nested before/after components"
                                  "depends on global state"
                                  "a"
                                  "b"]
                        :description "is a test"
                        :pending-reason "nothing has been asserted"}
                   #::s{:type ::s/context-finished
                        :subject "b"}
                   #::s{:type ::s/context-finished
                        :subject "a"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context with multiple nested before/after components"
                                  "depends on global state"]
                        :description "runs the before/after components in the specified order"}
                   #::s{:type ::s/context-finished
                        :subject "depends on global state"}
                   #::s{:type ::s/context-finished
                        :subject "a context with multiple nested before/after components"}
                   #::s{:type ::s/suite-finished}]}
  (c/let [state (atom [])
          reset-state (fn []
                        (swap! state empty))
          update-state (fn [n]
                         (fn []
                           (swap! state conj n)))]
    (c/context "depends on global state"
      (c/after-all reset-state)

      (c/context "a"
        (c/before-all
          (update-state 0))

        (c/before-each
          (update-state 2))

        (c/after-each
          (update-state 5))

        (c/after-all
          (update-state 7))

        (c/context "b"
          (c/before-all
            (update-state 1))

          (c/before-each
            (update-state 3))

          (c/after-each
            (update-state 4))

          (c/after-all
            (update-state 6))

          (c/it "is a test"
            (fn [assert-that]))))

      (c/it "runs the before/after components in the specified order"
        (fn [assert-that]
          (assert-that @state
                       (c/is [0 1 2 3 4 5 6 7])))))))

(c/describe "a context containing before/after components within a let"
  {:tags #{:fixture}
   ::r/event-data [#::r{:type ::r/suite-started}
                   #::r{:type ::r/context-started
                        :subject "a context containing before/after components within a let"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "passes"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "passes"}
                   #::r{:type ::r/test-started
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :description "passes"}
                   #::r{:type ::r/assertion-passed}
                   #::r{:type ::r/test-finished
                        :description "passes"}
                   #::r{:type ::r/context-finished
                        :subject "a context containing before/after components within a let"}
                   #::r{:type ::r/suite-finished}]
   ::s/event-data [#::s{:type ::s/suite-started}
                   #::s{:type ::s/context-started
                        :subject "a context containing before/after components within a let"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context containing before/after components within a let"]
                        :description "passes"}
                   #::s{:type ::s/test-passed
                        :namespace-name "coconut.v1.test-namespaces.fixtures"
                        :context ["a context containing before/after components within a let"]
                        :description "passes"}
                   #::s{:type ::s/context-finished
                        :subject "a context containing before/after components within a let"}
                   #::s{:type ::s/suite-finished}]}
  (c/let [n (atom 0)
          increment (fn []
                      (swap! n inc))
          reset (fn []
                  (reset! n 0))]
    (c/before-all increment)
    (c/before-each increment)
    (c/after-each increment)
    (c/after-all reset)

    (c/it "passes"
      (fn [assert-that]
        (assert-that @n (c/is 2))))

    (c/it "passes"
      (fn [assert-that]
        (assert-that @n (c/is 4))))))
