(ns coconut.v1.matchers
  #?(:cljs (:require-macros [coconut.v1.matchers]))
  (:require
    [clojure.set :as cs]
    [coconut.v1.platform :as platform]
    ))

(defprotocol IMatcher
  (evaluate-matcher [matcher actual]))

(extend-protocol IMatcher
  clojure.lang.IFn
  (evaluate-matcher [matcher actual]
    (matcher actual)))

(defn create-equality-matcher
  ([expected]
   (fn [actual]
     (when-not (= expected actual)
       {:expected (pr-str expected)
        :actual (pr-str actual)}))))

(defn create-negated-equality-matcher
  ([expected]
   (fn [actual]
     (when (= expected actual)
       {:expected (str "something other than " (pr-str expected))
        :actual (pr-str actual)}))))

(defn create-collection-membership-matcher
  ([value]
   (fn [collection]
     (when-not (contains? (set collection) value)
       {:expected (str "a collection containing " (pr-str value))
        :actual (pr-str collection)}))))

(defn create-subset-matcher
  ([elements]
   (fn [coll]
     (when-not (cs/subset? (set elements) (set coll))
       {:expected (str "a collection containing " (pr-str elements))
        :actual (pr-str coll)}))))

#?(:clj
(defmacro create-satisfies-predicate-matcher
  ([predicate]
   `(fn [value#]
      (when-not (~predicate value#)
        {:expected (platform/format "a value which satisfies `%s`"
                                    (pr-str '~predicate))
         :actual (pr-str value#)})))))

(defn create-throws-exception-matcher
  ([]
   (create-throws-exception-matcher platform/throwable-type))
  ([exception-class]
   (create-throws-exception-matcher exception-class nil))
  ([exception-class exception-message]
   (fn [f]
     (try (f)
          {:expected (str "a " (platform/exception-class-name exception-class) " to be thrown")
           :actual "no exception"}
          (catch #?(:clj Throwable :cljs :default) t
            (if-not (instance? exception-class t)
              {:expected (str "a " (platform/exception-class-name exception-class) " to be thrown")
               :actual (platform/exception-class-name t)}
              (when exception-message
                (when (not= (platform/exception-message t) exception-message)
                  {:expected (str "an exception message of " (pr-str exception-message))
                   :actual (pr-str (platform/exception-message t))}))))))))

(defn create-regex-matcher
  ([regex]
   (fn [string]
     (when-not (and (string? string)
                    (re-matches regex string))
       {:expected (platform/format "a string matching %s"
                                   (pr-str regex))
        :actual (pr-str string)}))))
