(ns coconut.v1.summarizing-test
  #?(:cljs (:require-macros [cljs.core.async.macros :as async]))
  (:require
    [clojure.core.async :as async :refer [<!]]
    [coconut.v1.core :as c]
    [coconut.v1.query :as query]
    [coconut.v1.running :as running]
    [coconut.v1.summarizing :as summarizing]
    [coconut.v1.platform :as platform]
    [coconut.v1.test-helpers :as helpers]
    [coconut.v1.test-namespaces.fixtures]
    ))

(defn summarize
  ([component]
   (summarizing/summarize (running/run (query/return component)))))

(defn summarize-fixtures
  ([]
   (->> (helpers/load-fixtures)
        (query/return-all)
        (running/run)
        (summarizing/summarize)
        (async/into []))))

(def number-of-events
  (helpers/number-of-events-function ::summarizing/type))

(c/describe `summarizing/summarize
  {:asynchronous {:timeout 250}}

  (c/for [component (helpers/load-fixtures)]
    (c/it (str "emits events for "
               (or (::c/subject component)
                   (::c/description component)))
      (fn [assert-that done]
        (async/go
          (let [event-channel (summarize component)]
            (doseq [data (::summarizing/event-data component)]
              (assert-that (<! event-channel)
                           (c/contains data)))
            (assert-that (<! event-channel)
                         (c/is nil))
            (done))))))

  (c/let [fixture-events (helpers/go->promise-channel (summarize-fixtures))]
    (c/it "includes the definition line number in all test outcome events"
      (fn [assert-that done]
        (async/go
          (let [line-numbers (into (vector)
                                   (comp (filter (comp (hash-set ::summarizing/test-passed
                                                                 ::summarizing/test-failed
                                                                 ::summarizing/test-threw-exception
                                                                 ::summarizing/test-timed-out
                                                                 ::summarizing/test-pending)
                                                       ::summarizing/type))
                                         (map ::summarizing/definition-line-number))
                                   (<! fixture-events))]
            (assert-that line-numbers
                         (c/satisfies (partial every? number?)))
            (done)))))

    (c/it "includes the definition line number in all test outcome events"
      (fn [assert-that done]
        (async/go
          (let [line-numbers (into (vector)
                                   (comp (filter (comp (hash-set ::summarizing/test-passed
                                                                 ::summarizing/test-failed
                                                                 ::summarizing/test-threw-exception
                                                                 ::summarizing/test-timed-out
                                                                 ::summarizing/test-pending)
                                                       ::summarizing/type))
                                         (map ::summarizing/definition-line-number))
                                   (<! fixture-events))]
            (assert-that line-numbers
                         (c/satisfies (partial every? number?)))
            (done)))))

    (c/it "includes the total number of tests in the suite started event"
      (fn [assert-that done]
        (async/go
          (let [events (<! fixture-events)]
            (assert-that (::summarizing/total-number-of-tests (first events))
                         (c/is (+ (number-of-events events ::summarizing/test-passed)
                                  (number-of-events events ::summarizing/test-failed)
                                  (number-of-events events ::summarizing/test-pending)
                                  (number-of-events events ::summarizing/test-threw-exception)
                                  (number-of-events events ::summarizing/test-timed-out))))
            (done)))))

    (c/it "includes the suite duration in the suite started event"
      (fn [assert-that done]
        (async/go
          (let [events (<! fixture-events)]
            (assert-that (::summarizing/duration-in-milliseconds (last events))
                         (c/satisfies number?))
            (done)))))))
