(ns coconut.v1.core-test
  (:require
    [coconut.v1.core :as c]
    [coconut.v1.test-helpers :as helpers]
    [coconut.v1.platform :as platform]
    [coconut.v1.query :as query]
    [coconut.v1.test-namespaces.core :as test-namespaces.core]
    ))

(defn find-component-with-unique-tag
  ([tag]
   (helpers/find-component-with-unique-tag 'coconut.v1.test-namespaces.core
                                           tag)))

(defn number-of-before-all-components
  ([context]
   (count (::c/components (::c/before-all context)))))

(defn number-of-before-each-components
  ([context]
   (count (::c/components (::c/before-each context)))))

(defn number-of-after-each-components
  ([context]
   (count (::c/components (::c/after-each context)))))

(defn number-of-after-all-components
  ([context]
   (count (::c/components (::c/after-all context)))))

(defn number-of-context-components
  ([context]
   (count (into (vector)
                (filter (comp (partial = ::c/context)
                              ::c/component-type))
                (::c/components (::c/sub-components context))))))

(defn number-of-test-components
  ([context]
   (count (into (vector)
                (filter (comp (partial = ::c/context)
                              ::c/component-type))
                (::c/components (::c/sub-components context))))))

(defn total-number-of-tests
  ([component]
   (::query/total-number-of-tests (query/return component))))

(def uuid-regex
  #".{8}-.{4}-.{4}-.{4}-.{12}")

(c/describe (platform/format "%s and %s" `c/describe `c/context)
  (c/let [component (find-component-with-unique-tag :3953659b-f974-4aeb-86d3-1e60d3405dfa)]

    (c/it "defines a context component"
      (fn [assert-that]
        (assert-that (::c/component-type component)
                     (c/is ::c/context))))

    (c/it "includes a unique id"
      (fn [assert-that]
        (assert-that (::c/id component)
                     (c/matches uuid-regex))))

    (c/it "includes the file path where it is defined"
      (fn [assert-that]
        #?(:clj
           (assert-that (::c/file-name component)
                        (c/matches #"coconut/v1/test_namespaces/core\.cljc")))
        #?(:cljs
           (assert-that (::c/file-name component)
                        (c/is-not nil)))))

    (c/it "includes the name of the namespace where it was defined"
      (fn [assert-that]
        (assert-that (::c/namespace-name component)
                     (c/is "coconut.v1.test-namespaces.core"))))

    (c/it "includes the line number where the test was defined"
      (fn [assert-that]
        (assert-that (::c/definition-line-number component)
                     (c/is 6))))

    (c/it "includes the subject"
      (fn [assert-that]
        (assert-that (::c/subject component)
                     (c/is "+"))))

    (c/context "when no before/after components are provided"
      (c/it "contains an empty before all collection component"
        (fn [assert-that]
          (assert-that (number-of-before-all-components component)
                       (c/is 0))))

      (c/it "contains an empty before each collection component"
        (fn [assert-that]
          (assert-that (number-of-before-each-components component)
                       (c/is 0))))

      (c/it "contains an empty after each collection component"
        (fn [assert-that]
          (assert-that (number-of-after-each-components component)
                       (c/is 0))))

      (c/it "contains an empty after all collection component"
        (fn [assert-that]
          (assert-that (number-of-after-all-components component)
                       (c/is 0)))))

    (c/for [data [{:subject "when the before/after components are defined top-level"
                   :component (find-component-with-unique-tag :7386ca36-2747-4eee-b6fb-8e6270caf3cc)}
                  {:subject "when the before/after components are defined within a let"
                   :component (find-component-with-unique-tag :0fe50449-b0b5-4459-b139-37f05b6fe4c5)}
                  {:subject "when the before/after components are defined within a for containing a single iteration"
                   :component (find-component-with-unique-tag :d2daccd4-dbcb-4a91-a852-4a4d51677520)}]]

      (c/context (:subject data)
        (c/it "includes the component in the before all collection component"
          (fn [assert-that]
            (assert-that (number-of-before-all-components (:component data))
                         (c/is 1))))

        (c/it "includes the component in the before each collection component"
          (fn [assert-that]
            (assert-that (number-of-before-each-components (:component data))
                         (c/is 1))))

        (c/it "includes the component in the after each collection component"
          (fn [assert-that]
            (assert-that (number-of-after-each-components (:component data))
                         (c/is 1))))

        (c/it "includes the component in the after all collection component"
          (fn [assert-that]
            (assert-that (number-of-after-all-components (:component data))
                         (c/is 1))))))

    (c/it "includes all defined sub-contexts"
      (fn [assert-that]
        (let [component (find-component-with-unique-tag :830ae88c-3283-44d4-8ac9-c2c9e48e2909)]
          (assert-that (number-of-context-components component)
                       (c/is 3)))))

    (c/it "includes all defined tests"
      (fn [assert-that]
        (let [component (find-component-with-unique-tag :830ae88c-3283-44d4-8ac9-c2c9e48e2909)]
          (assert-that (number-of-test-components component)
                       (c/is 3)))))

    (c/context "when not marked asynchronous"
      (c/let [component (find-component-with-unique-tag :7c23fd0b-87c3-47c8-81b3-de8683e5d2ef)]

        (c/it "does not indicate whether it is asynchronous or not"
          (fn [assert-that]
            (assert-that (::c/asynchronous? component)
                         (c/is nil))))

        (c/it "does not specify a timeout"
          (fn [assert-that]
            (assert-that (::c/timeout-in-milliseconds component)
                         (c/is nil))))))

    (c/context "when marked asynchronous"
      (c/let [component (find-component-with-unique-tag :13e6c394-4a21-4413-adbb-2fb0077e69ff)]

        (c/it "indicates it is asynchronous"
          (fn [assert-that]
            (assert-that (::c/asynchronous? component)
                         (c/is true))))

        (c/it "includes the timeout duration in milliseconds"
          (fn [assert-that]
            (assert-that (::c/timeout-in-milliseconds component)
                         (c/is 250))))))

    (c/context "when explicitly marked not asynchronous"
      (c/let [component (find-component-with-unique-tag :ee722610-50b7-490c-afcb-45a6236e95fc)]

        (c/it "indicates it is not asynchronous"
          (fn [assert-that]
            (assert-that (::c/asynchronous? component)
                         (c/is false))))

        (c/it "does not specify a timeout"
          (fn [assert-that]
            (assert-that (::c/timeout-in-milliseconds component)
                         (c/is nil))))))

    (c/context "when marked pending with a boolean"
      (c/let [component (find-component-with-unique-tag :9dc702d3-582e-48b7-aba9-b16bd6724547)]

        (c/it "indicates it is pending"
          (fn [assert-that]
            (assert-that (::c/pending? component)
                         (c/is true))))

        (c/it "does not include a reason"
          (fn [assert-that]
            (assert-that (::c/pending-reason component)
                         (c/is nil))))))

    (c/context "when marked pending with a reason"
      (c/let [component (find-component-with-unique-tag :c4b16359-6954-4497-9f2b-246bae933015)]

        (c/it "indicates it is pending"
          (fn [assert-that]
            (assert-that (::c/pending? component)
                         (c/is true))))

        (c/it "includes the reason"
          (fn [assert-that]
            (assert-that (::c/pending-reason component)
                         (c/is "for some reason"))))))

    (c/context "when tags are specified"
      (c/let [component (find-component-with-unique-tag :6b76a442-7478-4029-8a44-a8ff363c9f0a)]

        (c/it "includes the tags"
          (fn [assert-that]
            (assert-that (::c/tags component)
                         (c/contains-value :foo))

            (assert-that (::c/tags component)
                         (c/contains-value :bar))))))

    (c/it "merges unknown options into the component"
      (fn [assert-that]
        (let [component (find-component-with-unique-tag :12d90faa-dda0-4c59-8c6c-94901e0cae92)]
          (assert-that (::test-namespaces.core/foo component)
                       (c/is "bar")))))))

(c/describe (platform/format "%s and %s" `c/deftest `c/it)
  (c/let [component (find-component-with-unique-tag :9d198027-a5d3-4f8e-8b5a-8e85489019b8)]

    (c/it "creates a test component"
      (fn [assert-that]
        (assert-that (::c/component-type component)
                     (c/is ::c/test))))

    (c/it "includes a unique id"
      (fn [assert-that]
        (assert-that (::c/id component)
                     (c/matches uuid-regex))))

    (c/it "includes the file path where it is defined"
      (fn [assert-that]
        #?(:clj
           (assert-that (::c/file-name component)
                        (c/matches #"coconut/v1/test_namespaces/core\.cljc")))
        #?(:cljs
           (assert-that (::c/file-name component)
                        (c/is-not nil)))))

    (c/it "includes the name of the namespace where it was defined"
      (fn [assert-that]
        (assert-that (::c/namespace-name component)
                     (c/is "coconut.v1.test-namespaces.core"))))

    (c/it "includes the line number where the test was defined"
      (fn [assert-that]
        (assert-that (::c/definition-line-number component)
                     (c/is 97))))

    (c/it "includes the description"
      (fn [assert-that]
        (assert-that (::c/description component)
                     (c/is "+ returns the sum of two numbers"))))

    (c/context "when not marked asynchronous"
      (c/let [component (find-component-with-unique-tag :53104943-7b89-40d7-a441-360c79e2d30a)]

        (c/it "does not indicate whether it is asynchronous or not"
          (fn [assert-that]
            (assert-that (::c/asynchronous? component)
                         (c/is nil))))

        (c/it "does not specify a timeout"
          (fn [assert-that]
            (assert-that (::c/timeout-in-milliseconds component)
                         (c/is nil))))))

    (c/context "when marked asynchronous"
      (c/let [component (find-component-with-unique-tag :70141b5f-22d9-4275-b2ec-7498683423ea)]

        (c/it "indicates it is asynchronous"
          (fn [assert-that]
            (assert-that (::c/asynchronous? component)
                         (c/is true))))

        (c/it "includes the timeout duration in milliseconds"
          (fn [assert-that]
            (assert-that (::c/timeout-in-milliseconds component)
                         (c/is 250))))))

    (c/context "when explicitly marked not asynchronous"
      (c/let [component (find-component-with-unique-tag :a39c7b30-1935-49eb-bbf1-0d8e7eac57e5)]

        (c/it "indicates it is not asynchronous"
          (fn [assert-that]
            (assert-that (::c/asynchronous? component)
                         (c/is false))))

        (c/it "does not specify a timeout"
          (fn [assert-that]
            (assert-that (::c/timeout-in-milliseconds component)
                         (c/is nil))))))

    (c/context "when marked pending with a boolean"
      (c/let [component (find-component-with-unique-tag :e65d6526-2bea-420c-aac1-f6e1f3cfddcf)]

        (c/it "indicates it is pending"
          (fn [assert-that]
            (assert-that (::c/pending? component)
                         (c/is true))))

        (c/it "does not include a reason"
          (fn [assert-that]
            (assert-that (::c/pending-reason component)
                         (c/is nil))))))

    (c/context "when marked pending with a reason"
      (c/let [component (find-component-with-unique-tag :6efead89-6dd2-4477-975a-4f4ca87e402c)]

        (c/it "indicates it is pending"
          (fn [assert-that]
            (assert-that (::c/pending? component)
                         (c/is true))))

        (c/it "includes the reason"
          (fn [assert-that]
            (assert-that (::c/pending-reason component)
                         (c/is "for some reason"))))))

    (c/context "when tags are specified"
      (c/let [component (find-component-with-unique-tag :6d152479-2ac0-49f5-89f5-e0ffb9818850)]

        (c/it "includes the tags"
          (fn [assert-that]
            (assert-that (::c/tags component)
                         (c/contains-value :foo))

            (assert-that (::c/tags component)
                         (c/contains-value :bar))))))

    (c/it "merges unknown options into the component"
      (fn [assert-that]
        (let [component (find-component-with-unique-tag :bce4c8c7-abcc-46d3-bce1-ad62bb6c43e3)]
          (assert-that (::test-namespaces.core/foo component)
                       (c/is "bar")))))))

(c/describe `c/for
  (c/it "accepts zero tests"
    (fn [assert-that]
      (let [component (c/context "should contain 6 tests"
                        (c/for [x [0 1 2]]))]
        (assert-that (total-number-of-tests component)
                     (c/is 0)))))

  (c/it "accepts one test"
    (fn [assert-that]
      (let [component (c/context "should contain 6 tests"
                        (c/for [x [0 1 2]]
                          (c/it "has access to x"
                            (fn [_] x))))]
        (assert-that (total-number-of-tests component)
                     (c/is 3)))))

  (c/it "accepts multiple tests"
    (fn [assert-that]
      (let [component (c/context "should contain 6 tests"
                        (c/for [x [0 1 2]]
                          (c/it "has access to x"
                            (fn [_] x))
                          (c/it "has access to x"
                            (fn [_] x))))]
        (assert-that (total-number-of-tests component)
                     (c/is 6))))))

(c/describe `c/let
  (c/it "accepts zero tests"
    (fn [assert-that]
      (let [component (c/context "should contain 6 tests"
                        (c/let [x 0]))]
        (assert-that (total-number-of-tests component)
                     (c/is 0)))))

  (c/it "accepts one test"
    (fn [assert-that]
      (let [component (c/context "should contain 6 tests"
                        (c/let [x 0]
                          (c/it "has access to x"
                            (fn [_] x))))]
        (assert-that (total-number-of-tests component)
                     (c/is 1)))))

  (c/it "accepts multiple tests"
    (fn [assert-that]
      (let [component (c/context "should contain 6 tests"
                        (c/let [x 0]
                          (c/it "has access to x"
                            (fn [_] x))
                          (c/it "has access to x"
                            (fn [_] x))))]
        (assert-that (total-number-of-tests component)
                     (c/is 2))))))
