(ns coconut.v1.summarizing
  #?(:cljs (:require-macros [cljs.core.async.macros :as async]))
  (:require
    [clojure.core.async :as async :refer [<! >!]]
    [coconut.v1.running :as running]
    [coconut.v1.platform :as platform]
    ))

(defmulti ^{:private true} summarize-test-event
  (fn [summarized-event-channel test-execution-state event]
    [(::current-status test-execution-state) (::running/type event)]))

(defmethod ^{:private true} summarize-test-event
  [::not-started ::running/test-started]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (merge test-execution-state
            #::{:namespace-name (::running/namespace-name event)
                :definition-line-number (::running/definition-line-number event)
                :current-status ::nothing-asserted}))))

(defmethod ^{:private true} summarize-test-event
  [::nothing-asserted ::running/assertion-passed]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (merge test-execution-state
            #::{:current-status ::passing}))))

(defmethod ^{:private true} summarize-test-event
  [::nothing-asserted ::running/assertion-failed]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (merge test-execution-state
            #::{:current-status ::failing
                :relevant-assertion-failed-event event}))))

(defmethod ^{:private true} summarize-test-event
  [::nothing-asserted ::running/test-timed-out]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::test-timed-out
             :context (::context test-execution-state)
             :namespace-name (::namespace-name test-execution-state)
             :definition-line-number (::definition-line-number test-execution-state)
             :description (::running/description event)})
     (merge test-execution-state
            #::{:current-status ::finished}))))

(defmethod ^{:private true} summarize-test-event
  [::nothing-asserted ::running/test-threw-exception]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::test-threw-exception
             :namespace-name (::namespace-name test-execution-state)
             :definition-line-number (::definition-line-number test-execution-state)
             :context (::context test-execution-state)
             :description (::running/description event)
             :exception (::running/exception event)})
     (merge test-execution-state
            #::{:current-status ::finished}))))

(defmethod ^{:private true} summarize-test-event
  [::nothing-asserted ::running/test-finished]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::test-pending
             :definition-line-number (::definition-line-number test-execution-state)
             :context (::context test-execution-state)
             :description (::running/description event)
             :pending-reason "nothing has been asserted"})
     (merge test-execution-state
            #::{:current-status ::finished}))))

(defmethod ^{:private true} summarize-test-event
  [::failing ::running/assertion-passed]
  ([summarized-event-channel test-execution-state event]
   (async/go test-execution-state)))

(defmethod ^{:private true} summarize-test-event
  [::failing ::running/assertion-failed]
  ([summarized-event-channel test-execution-state event]
   (async/go test-execution-state)))

(defmethod ^{:private true} summarize-test-event
  [::failing ::running/test-timed-out]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::test-timed-out
             :context (::context test-execution-state)
             :namespace-name (::namespace-name test-execution-state)
             :definition-line-number (::definition-line-number test-execution-state)
             :description (::running/description event)})
     (merge test-execution-state
            #::{:current-status ::finished}))))

(defmethod ^{:private true} summarize-test-event
  [::failing ::running/test-threw-exception]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::test-failed
             :namespace-name (::namespace-name test-execution-state)
             :definition-line-number (::definition-line-number test-execution-state)
             :context (::context test-execution-state)
             :description (::running/description event)
             :expected (::running/expected (::relevant-assertion-failed-event test-execution-state))
             :actual (::running/actual (::relevant-assertion-failed-event test-execution-state))})
     (merge test-execution-state
            #::{:current-status ::finished}))))

(defmethod ^{:private true} summarize-test-event
  [::failing ::running/test-finished]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::test-failed
             :namespace-name (::namespace-name test-execution-state)
             :definition-line-number (::definition-line-number test-execution-state)
             :context (::context test-execution-state)
             :description (::running/description event)
             :expected (::running/expected (::relevant-assertion-failed-event test-execution-state))
             :actual (::running/actual (::relevant-assertion-failed-event test-execution-state))})
     (merge test-execution-state
            #::{:current-status ::finished}))))

(defmethod ^{:private true} summarize-test-event
  [::passing ::running/assertion-passed]
  ([summarized-event-channel test-execution-state event]
   (async/go test-execution-state)))

(defmethod ^{:private true} summarize-test-event
  [::passing ::running/assertion-failed]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (merge test-execution-state
            #::{:current-status ::failing
                :relevant-assertion-failed-event event}))))

(defmethod ^{:private true} summarize-test-event
  [::passing ::running/test-timed-out]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::test-timed-out
             :context (::context test-execution-state)
             :namespace-name (::namespace-name test-execution-state)
             :definition-line-number (::definition-line-number test-execution-state)
             :description (::running/description event)})
     (merge test-execution-state
            #::{:current-status ::finished}))))

(defmethod ^{:private true} summarize-test-event
  [::passing ::running/test-threw-exception]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::test-threw-exception
             :namespace-name (::namespace-name test-execution-state)
             :definition-line-number (::definition-line-number test-execution-state)
             :context (::context test-execution-state)
             :description (::running/description event)
             :exception (::running/exception event)})
     (merge test-execution-state
            #::{:current-status ::finished}))))

(defmethod ^{:private true} summarize-test-event
  [::passing ::running/test-finished]
  ([summarized-event-channel test-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::test-passed
             :namespace-name (::namespace-name test-execution-state)
             :definition-line-number (::definition-line-number test-execution-state)
             :context (::context test-execution-state)
             :description (::running/description event)})
     (merge test-execution-state
            #::{:current-status ::finished}))))

(defmulti ^{:private true} summarize-event
  (fn [summarized-event-channel event-channel suite-execution-state event]
    (::running/type event)))

(defmethod ^{:private true} summarize-event
  ::running/suite-started
  ([summarized-event-channel event-channel suite-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::suite-started
             :total-number-of-tests (::running/total-number-of-tests event)})
     (merge suite-execution-state
            #::{:suite-started-at (::running/current-time-millis event)}))))

(defmethod ^{:private true} summarize-event
  ::running/context-started
  ([summarized-event-channel event-channel suite-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::context-started
             :subject (::running/subject event)})
     (update suite-execution-state
             ::context
             conj
             (::running/subject event)))))

(defmethod ^{:private true} summarize-event
  ::running/test-marked-pending
  ([summarized-event-channel event-channel suite-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::test-pending
             :definition-line-number (::running/definition-line-number event)
             :context (::context suite-execution-state)
             :description (::running/description event)
             :pending-reason (::running/pending-reason event)
             :duration-in-milliseconds 0})
     suite-execution-state)))

(defmethod ^{:private true} summarize-event
  ::running/test-started
  ([summarized-event-channel event-channel suite-execution-state event]
   (async/go
     (loop [state #::{:current-status ::not-started
                      :relevant-assertion-failed-event nil
                      :context (::context suite-execution-state)}
            event event]
       (let [updated-state (<! (summarize-test-event summarized-event-channel state event))]
         (when-not (= ::finished (::current-status updated-state))
           (recur updated-state
                  (or (<! event-channel)
                      (throw (platform/exception "unexpected sequence of running events!")))))))
     suite-execution-state)))

(defmethod ^{:private true} summarize-event
  ::running/context-finished
  ([summarized-event-channel event-channel suite-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::context-finished
             :subject (::running/subject event)})
     (update suite-execution-state
             ::context
             pop))))

(defmethod ^{:private true} summarize-event
  ::running/suite-finished
  ([summarized-event-channel event-channel suite-execution-state event]
   (async/go
     (>! summarized-event-channel
         #::{:type ::suite-finished
             :duration-in-milliseconds (- (::running/current-time-millis event)
                                          (::suite-started-at suite-execution-state))})
     suite-execution-state)))

(defn summarize
  "Given a core.async channel of test events, returns a new channel of
  events which indicates the outcome of the suite and its contained contexts
  and tests."
  ([event-channel]
   (let [summarized-event-channel (async/chan 250)]
     (async/go
       (loop [suite-execution-state #::{:suite-started-at nil
                                        :context []}]
         (when-let [event (<! event-channel)]
           (recur (<! (summarize-event summarized-event-channel event-channel suite-execution-state event)))))
       (async/close! summarized-event-channel))
     summarized-event-channel)))
