(ns coconut.v1.karma
  (:require-macros [cljs.core.async.macros :as async])
  (:require
    [clojure.core.async :as async]
    [coconut.v1.summarizing :as summarizing]
    [coconut.v1.platform :as platform]
    ))

(def ^{:private true} global-karma
  (aget js/window "__karma__"))

(def ^{:private true} coverage
  (aget js/window "__coverage__"))

(defprotocol IKarma
  (info [this data])
  (result [this data])
  (complete [this data]))

(defrecord Karma [instance]
  IKarma
  (info [this data]
    (.info instance
           (clj->js data)))

  (result [this data]
    (.result instance
             (clj->js data)))

  (complete [this data]
    (.complete instance
               (clj->js data))))

(defmulti ^{:private true} report-event
  (fn [karma event]
    (::summarizing/type event)))

(defmethod ^{:private true} report-event
  ::summarizing/suite-started
  ([karma event]
   (info karma
         {:total (::summarizing/total-number-of-tests event)})))

(defmethod ^{:private true} report-event
  ::summarizing/test-pending
  ([karma event]
   (result karma
           {:description (::summarizing/description event)
            :suite (::summarizing/context event)
            :success true
            :skipped true
            :time (::summarizing/duration-in-milliseconds event)
            :log [(str "PENDING -- " (::summarizing/pending-reason event))]})))

(defmethod ^{:private true} report-event
  ::summarizing/test-passed
  ([karma event]
   (result karma
           {:description (::summarizing/description event)
            :suite (::summarizing/context event)
            :success true
            :skipped false
            :time (::summarizing/duration-in-milliseconds event)
            :log []})))

(defmethod ^{:private true} report-event
  ::summarizing/test-failed
  ([karma event]
   (result karma
           {:description (::summarizing/description event)
            :suite (::summarizing/context event)
            :success false
            :skipped false
            :time (::summarizing/duration-in-milliseconds event)
            :log [(str "expected: " (::summarizing/expected event) \newline
                       "  actual: " (::summarizing/actual event))]})))

(defmethod ^{:private true} report-event
  ::summarizing/test-timed-out
  ([karma event]
   (result karma
           {:description (::summarizing/description event)
            :suite (::summarizing/context event)
            :success false
            :skipped false
            :time (::summarizing/duration-in-milliseconds event)
            :log ["TIMEOUT"]})))

(defmethod ^{:private true} report-event
  ::summarizing/test-threw-exception
  ([karma event]
   (result karma
           {:description (::summarizing/description event)
            :suite (::summarizing/context event)
            :success false
            :skipped false
            :time (::summarizing/duration-in-milliseconds event)
            :log [(apply str
                         (interpose \newline
                                    (list* (platform/exception-class-name (::summarizing/exception event))
                                           (platform/exception-message (::summarizing/exception event))
                                           (platform/exception-stack-trace (::summarizing/exception event)))))]})))

(defmethod ^{:private true} report-event
  ::summarizing/suite-finished
  ([karma event]
   (complete karma
             {:coverage coverage})))

(defmethod ^{:private true} report-event
  :default
  ([_ _]))

(def karma?
  "Returns true when the __karma__ global variable is present. Indicates
  the code is probably being executed within karma."
  (some? global-karma))

(defn set-karma-start-function!
  "Sets the start function on the global __karma__ instance. Karma
  executes this function to start the test run."
  ([f]
   (aset global-karma "start" f)))

(defn report
  "Given a core.async channel of summarized test events, reports
  the outcomes of the suite/tests to Karma."
  ([event-channel]
   (report event-channel (->Karma global-karma)))
  ([event-channel karma]
   (async/go
     (loop []
       (when-let [event (async/<! event-channel)]
         (report-event karma event)
         (recur))))))
