(ns via.core
  (:require [via.endpoint :as via]
            [via.subs :as vs]
            [via.events :as ve]
            [via.defaults :as defaults]
            [via.util.promise :as p]
            #?(:cljs [via.re-frame :as rf])
            [utilis.map :refer [compact]]
            [signum.fx :as sfx]
            [signum.subs :as ss]))

(defn subscribe
  ([query]
   (let [endpoint (via/first-endpoint)]
     (subscribe endpoint (via/first-peer endpoint) query nil)))
  ([query default]
   (let [endpoint (via/first-endpoint)]
     (subscribe endpoint (via/first-peer endpoint) query default)))
  ([endpoint peer-id query]
   (subscribe endpoint peer-id query nil))
  ([endpoint peer-id query default]
   #?(:cljs (rf/subscribe endpoint peer-id query default)
      :clj (vs/subscribe endpoint peer-id query default))))

(defn dispatch
  "Dispatch `event` to `peer-id` through `endpoint`."
  ([event]
   (let [endpoint (via/first-endpoint)]
     (dispatch endpoint (via/first-peer endpoint) event)))
  ([peer-id event]
   (let [endpoint (via/first-endpoint)]
     (dispatch endpoint peer-id event)))
  ([endpoint peer-id event]
   #?(:cljs (rf/dispatch endpoint peer-id event)
      :clj (ve/dispatch endpoint peer-id event))))

(defn invoke
  "Invoke the handler for `event` on `peer-id` through `endpoint`,
  with a response from the peer expected, which can be handled using
  promise syntax."
  ([event]
   (let [endpoint (via/first-endpoint)]
     (invoke endpoint (via/first-peer endpoint) event nil)))
  ([event options]
   (let [endpoint (via/first-endpoint)]
     (invoke endpoint (via/first-peer endpoint) event options)))
  ([endpoint peer-id event]
   (invoke endpoint peer-id event nil))
  ([endpoint peer-id event options]
   #?(:cljs (rf/invoke endpoint peer-id event options)
      :clj (ve/invoke endpoint peer-id event options))))

(sfx/reg-fx
 :via/dispatch
 (fn [_ event-or-map]
   (cond
     (sequential? event-or-map)
     (dispatch (vec event-or-map))

     (map? event-or-map)
     (let [{:keys [peer-id endpoint event]} event-or-map]
       (cond
         (and endpoint peer-id)
         (dispatch endpoint peer-id event)

         peer-id
         (dispatch peer-id event)

         :else (dispatch event)))
     :else (throw (ex-info "Argument to :via/dispatch must be an event vector or a map of options."
                           {:event-or-map event-or-map})))))

(sfx/reg-fx
 :via/invoke
 (fn [_ {:keys [event
               timeout
               on-success
               on-failure
               on-timeout
               endpoint
               peer-id]}]
   (let [endpoint (or endpoint (via/first-endpoint))]
     (invoke endpoint
             (or peer-id (via/first-peer endpoint))
             event
             (compact
              {:timeout timeout
               :on-success on-success
               :on-failure on-failure
               :on-timeout on-timeout})))))
