;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.subs
  (:refer-clojure :exclude [proxy subs])
  (:require
   #?(:cljs [utilis.js :as j])
   [distantia.core :refer [diff patch]]
   [fluxus.lock :refer [with-lock]]
   [fluxus.promise :as p]
   [signum.events :as se]
   [signum.signal :refer [alter! signal]]
   [signum.subs :as ss]
   [spectator.log :as log]
   [tempus.duration :as td]
   [via.events :as ve]
   [via.service :as vs])
  #?(:clj (:import
           [java.util.concurrent.locks ReentrantLock]
           [via.proxy Proxy])))

(def default-timeout (td/minutes 1))

(defonce ^:private local-signals (atom {})) ; watched for remote side
(defonce ^:private signal-proxys (atom {})) ; updated by remote side

(declare dispose-local)

(se/reg-event
 :-vs/s
 (fn [{:keys [service proxy] :as _context} [_ {[query-id & _ :as query] :q callback-ev :c sn :sn}]]
   (let [send-value (fn [m]
                      (log/trace [:via.sub/send-value query callback-ev m])
                      (-> (ve/invoke proxy (conj (vec callback-ev) m)
                                     {:timeout default-timeout})
                          (p/then
                            (fn [{:keys [status] :as _response}]
                              (when (not= 200 status)
                                (log/trace [:via.sub/disposing (:id proxy) query])
                                (dispose-local proxy query))))
                          (p/catch
                            (fn [_]
                              (log/error [:via.sub/send-value :timeout (:id proxy) query])
                              ;; TODO: Handle timeout
                              ))))]
     (if-not (and (ss/sub? query-id) (vs/exported? service :sub query-id))
       (do
         (log/error [:via.sub/unknown (:id proxy) query])
         {:via/reply {:status 400
                      :body {:e :via.sub/unknown
                             :q query}}})
       (if-let [watch (get @local-signals [proxy query])]
         (let [{last-sn :sn signal :signal} watch]
           (log/trace [:via.sub/last-sn query last-sn])
           (when (not= @last-sn sn)
             (send-value {:sn @last-sn :c [:v @signal]}))
           {:via/reply {:status 200}})
         (let [signal (binding [ss/*context* {:coeffects
                                              (merge (:coeffects proxy)
                                                     {:service service
                                                      :proxy proxy})}]
                        (ss/subscribe query))
               watch-key (:id proxy)
               sn (atom -1)]
           (add-watch
            signal watch-key
            (fn [_ _ old new]
              (when (not= old new)
                (let [sn (swap! sn inc)
                      value (if (instance? #?(:clj Throwable :cljs :default) new)
                              {:sn sn
                               :e (let [ex-data (ex-data new)]
                                    (cond-> {:m #?(:clj  (.getMessage ^Throwable new)
                                                   :cljs (j/get new :message))}
                                      ex-data (assoc :d ex-data)))}
                              {:sn sn
                               :c (if (or (and (map? new) (map? old))
                                          (and (vector? new) (vector? old)))
                                    [:p (diff old new)]
                                    [:v new])})]
                  (send-value value)))))
           (swap! local-signals
                  assoc [proxy query]
                  {:signal signal
                   :sn sn
                   :dispose-fn #(remove-watch signal watch-key)})
           (add-watch proxy watch-key
                      (fn [_ _ _ state]
                        (when (= :released state)
                          (dispose-local proxy query))))
           {:via/reply {:status 200}}))))))

(se/reg-event
 :-vs/d
 (fn [{:keys [proxy] :as _context} [_ {query :q}]]
   (dispose-local proxy query)
   {:via/reply {:status 200}}))

(declare remote-subscribe)

(se/reg-event
 :-vss/u
 (fn [{:keys [proxy] :as _context} [_ signal-id {error :e :as message}]]
   (let [{:keys [query signal] :as signal-proxy} (get @signal-proxys signal-id)]
     (if error
       (log/error [:via.sub/remote-error (:id proxy) query error])
       (if-not signal-proxy
         (do
           (log/error [:via.sub/unknown (:id proxy) {:signal-id signal-id
                                                     :message message}])
           {:via/reply {:status 400
                        :body {:e :unknown}}})
         (let [{last-sn :sn} signal-proxy
               {:keys [sn c]} message
               set-value (fn []
                           (reset! last-sn sn)
                           (alter! signal (fn [value]
                                            (if (= :v (first c))
                                              (second c)
                                              (patch value (second c)))))
                           {:via/reply {:status 200}})]
           (if (or (= :v (first c)) (= sn (inc @last-sn)))
             (set-value)
             (do (remote-subscribe proxy query signal-id)
                 {:via/reply {:status 400}}))))))))

(declare local-query reg-proxy-sub)

(defonce ^:private subscription-lock #?(:clj (ReentrantLock.) :cljs nil))

(defn subscribe
  ([^Proxy proxy query]
   (subscribe proxy query nil))
  ([^Proxy proxy query {:keys [timeout] :or {timeout default-timeout}}]
   (with-lock [subscription-lock]
     (let [[local-query-id & _ :as local-query] (local-query proxy query)]
       (log/trace [:via/subscribe (:id proxy) query local-query (ss/sub? local-query-id)])
       (when (not (ss/sub? local-query-id))
         (reg-proxy-sub proxy {:local-query local-query
                               :timeout timeout}))
       (ss/subscribe local-query)))))


;;; Private

(defn- remote-subscribe
  ([proxy query signal-id]
   (remote-subscribe proxy query signal-id default-timeout))
  ([proxy query signal-id timeout]
   (if-let [sn (get-in @signal-proxys [signal-id :sn])]
     (-> (ve/invoke proxy
                    [:-vs/s {:q query
                             :sn @sn
                             :c [:-vss/u signal-id]}]
                    {:timeout timeout})
         (p/then
           (fn [{:keys [status] :as response}]
             (if (= 200 status)
               (log/trace [:via.sub/remote-subscribed (:id proxy) query])
               (log/error [:via.sub/remote-error (:id proxy) response]))))
         (p/catch
           (fn [_]
             (log/error [:via.sub/remote-timeout (:id proxy) query signal-id])
             ;; TODO: Handle timeout
             )))
     (log/error [:via.sub/remote-subscribe (:id proxy) query signal-id :missing-proxy]))))

(defn- local-query
  [proxy query]
  (let [local-query-id [(first query) proxy]]
    (vec (cons local-query-id (rest query)))))

(defn- remote-query
  [query]
  (let [[query-id _] (first query)]
    (vec (cons query-id (rest query)))))

(defn- reg-proxy-sub
  [proxy {:keys [local-query timeout]}]
  (let [[local-query-id & _] local-query]
    (log/trace [:via/reg-sub :create (:id proxy) local-query])
    (ss/reg-sub
     local-query-id
     (fn [local-query]
       (try
         (let [signal (signal)
               signal-id (hash signal)
               remote-query (remote-query local-query)]
           (log/trace [:via/reg-sub :init (:id proxy) remote-query signal])
           (swap! signal-proxys
                  assoc signal-id {:query remote-query
                                   :signal signal
                                   :sn (atom -1)})
           (add-watch proxy signal-id
                      (fn [_ _ _ state]
                        (when (= :connected state)
                          (log/trace [:via/reg-sub :connected (:id proxy)])
                          (remote-subscribe proxy remote-query signal-id timeout))))
           [signal-id signal])
         (catch #?(:clj Throwable :cljs :default) e
           (log/error [:via/reg-sub :init (:id proxy)] e))))
     (fn [[signal-id _] local-query]
       (let [remote-query (remote-query local-query)]
         (log/trace [:via/reg-sub :dispose (:id proxy) remote-query])
         (remove-watch proxy signal-id)
         (ve/dispatch proxy [:-vs/d {:q remote-query}])
         (swap! signal-proxys dissoc signal-id)))
     (fn [[_ signal] _]
       @signal))))

(defn- dispose-local
  [proxy query]
  (log/trace [:via.sub/dispose-local (:id proxy) query])
  (when-let [dispose-fn (get-in @local-signals [[proxy query] :dispose-fn])]
    (dispose-fn))
  (swap! local-signals dissoc [proxy query])
  (let [[local-query-id & _] (local-query proxy query)]
    (ss/dereg-sub local-query-id)))
