;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.proxy
  (:refer-clojure :exclude [proxy send])
  (:require [fluxus.flow :as f]
            [fluxus.promise :as p]
            [tempus.duration :as td]
            [utilis.timer :as timer]
            [utilis.string :as ust]
            [spectator.log :as log]
            #?(:clj [clojure.pprint :refer [simple-dispatch]]))
  #?(:clj (:import [clojure.lang IObj IRef ILookup IMeta])))

#?(:cljs (set! *warn-on-infer* true))

(declare pr-proxy)

(deftype Proxy [id status token-fn encode decode connection
                requests coeffects meta-map]
  Object
  (toString
    [^Proxy this]
    (pr-proxy this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash)
    [_]
    (hash [:via/service id]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv)
    [^Proxy this other]
    (boolean
     (when (instance? Proxy other)
       (= (#?(:clj .id :cljs .-id) this)
          (#?(:clj .id :cljs .-id) ^Proxy other)))))

  ILookup
  #?@(:clj
      [(valAt
        [_this k default]
        (case k
          :id (clojure.core/deref id)
          :status (clojure.core/deref status)
          :coeffects (clojure.core/deref coeffects)
          default))
       (valAt
        [this k]
        (let [v (get this k ::not-found)]
          (if (= ::not-found v)
            (throw (ex-info ":via/proxy invalid key" {:key k}))
            v)))]
      :cljs
      [(-lookup
        [_this k default]
        (case k
          :id id
          :status (clojure.core/deref status)
          :coeffects (clojure.core/deref coeffects)
          default))
       (-lookup
        [this k]
        (let [v (-lookup this k ::not-found)]
          (if (= ::not-found v)
            (throw (ex-info ":via/proxy invalid key" {:key k}))
            v)))])

  #?(:clj IRef :cljs IWatchable)
  (#?(:clj addWatch :cljs -add-watch)
    [this watch-key watch-fn]
    (add-watch
     status watch-key
     (fn [_key _ref old-value new-value]
       (when (not= old-value new-value)
         (watch-fn watch-key this old-value new-value))))
    (when-let [status (clojure.core/deref status)]
      (watch-fn watch-key this nil status))
    this)
  (#?(:clj removeWatch :cljs -remove-watch)
    [this watch-key]
    (remove-watch status watch-key)
    this)

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Proxy. id status token-fn encode decode connection
            requests coeffects meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-proxy this)))]))

#?(:clj
   (defmethod print-method Proxy
     [^Proxy p w]
     (.write ^java.io.Writer w ^String (pr-proxy p))))

#?(:clj
   (defmethod simple-dispatch Proxy
     [^Proxy p]
     (print (pr-proxy p))))

(defn proxy
  [id token-fn encode decode connection-p]
  (Proxy. (atom id)
          (atom nil)
          token-fn
          encode
          decode
          (atom connection-p)
          (atom {})
          (atom nil)
          nil))

(defn token
  [^Proxy proxy]
  ((#?(:clj .token-fn :cljs .-token-fn) proxy)))

(defn send
  [^Proxy proxy message]
  (log/trace [:via/>? proxy message])
  (let [id (:id proxy)
        connection (#?(:clj .connection :cljs .-connection) proxy)
        encode (#?(:clj .encode :cljs .-encode) proxy)]
    (-> @connection
        (p/then
          (fn [connection]
            (log/trace [:via/>? @id connection message])
            (-> (f/put! connection (encode message))
                (p/then
                  (fn [v]
                    (log/trace [:via/> @id connection message v])))
                (p/catch
                  (fn [e]
                    (log/trace [:via/>! @id connection message] e)
                    ;; TODO: Handle error
                    ))))))))

(defn invoke
  ([^Proxy proxy message]
   (invoke proxy message {}))
  ([^Proxy proxy message {:keys [timeout]
                          :or {timeout (td/seconds 60)}}]
   (log/trace [:via/><? proxy message timeout])
   (let [id (:id proxy)
         reply (p/promise {:label (str "(invoke " id " " (pr-str message) ")")})
         reply-id (token proxy)
         requests (#?(:clj .requests :cljs .-requests) proxy)]
     (try
       (swap! requests
              assoc reply-id
              {:on-reply
               (fn [value]
                 (when-not (p/realized? reply)
                   (p/resolve! reply value)))
               :timer
               (timer/run-after
                (fn []
                  (log/trace [:via/><! :timeout proxy message])
                  (when-not (p/realized? reply)
                    (p/reject! reply :via.invoke/timeout))
                  (swap! requests dissoc reply-id))
                (td/into :milliseconds timeout))})
       (send proxy (assoc message :reply-id reply-id))
       (catch #?(:clj Throwable :cljs :default) e
         (log/error [:via/><? proxy] e)))
     reply)))


;;; Private

(defn- pr-proxy
  [^Proxy p]
  (ust/format
   (str "#<via/proxy@" #?(:clj "0x%x" :cljs "%s") "%s%s>")
   (hash p)
   (str "[" (:id p) "] ")
   (str (let [connection-p @(#?(:clj .connection :cljs .-connection) p)]
          (if (p/realized? connection-p)
            @connection-p
            "...")))
   (str ":requests " (vec (keys @(#?(:clj .requests :cljs .-requests) p))))))
