;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.schema
  #?(:cljs (:require-macros [via.schema]))
  (:require #?(:clj [schema.core :as s])
            #?(:clj [schema.util.parse :refer [parse-args]])
            [malli.core :as m]
            [malli.transform :as mt]
            [malli.error :as me]
            [malli.util :as mu]))

#?(:clj
   (def tx
     (mt/transformer
      (mt/strip-extra-keys-transformer)
      (mt/default-value-transformer))))

#?(:clj
   (defmacro >fn
     [& args]
     (let [{:keys [params body in-schema out-schema]} (parse-args args)
           in-schema-tuple (into [:tuple] in-schema)
           out-schema [:multi {:dispatch #(boolean
                                           (and (map? %)
                                                (contains? % :via/reply)))}
                       [true [:map
                              [:via/reply
                               [:map
                                [:body {:optional true} out-schema]
                                [:status :int]]]]]
                       [false out-schema]]]
       `(let [in-schema-tuple# ~in-schema-tuple
              out-schema# ~out-schema
              in-decoder# (m/decoder in-schema-tuple# tx)
              out-decoder# (m/decoder out-schema# tx)
              f# (s/>fn ~params
                   ~(into in-schema ['=> out-schema])
                   ~@body)]
          (fn [& args#]
            (let [args# (in-decoder# (vec args#))
                  result# (apply f# args#)]
              (if (:via/reply result#)
                (update-in result# [:via/reply :body] out-decoder#)
                (out-decoder# result#))))))))


;;; Schemas

(defn validate-throw
  [schema x]
  (let [valid (m/validate schema (m/decode schema x mt/strip-extra-keys-transformer))]
    (when (not valid)
      (throw (ex-info "Schema Validation Error"
                      {:value x
                       :explanation (-> schema
                                        (m/explain x)
                                        me/with-spell-checking
                                        me/humanize)})))
    x))

(def Some
  [:fn {:error/fn (fn [{:keys [value]} _]
                    "Value must not be nil.")}
   some?])

(def some-validator (m/validator Some))
(defn validate-some
  [& args]
  (when (not (every? some-validator args))
    (throw (ex-info "Schema Validation Error"
                    {:explanation {:message "Arguments must not be nil."
                                   :args args}}))))

(def ExportType
  [:enum :sub :event :namespace])

(def Address
  [:or
   :string
   [:map
    [:host :string]
    [:port int?]
    [:scheme :string]
    [:path :string]
    [:tls [:map
           [:server [:map
                     [:cert :string]
                     [:key :string]]]
           [:client [:map
                     [:ca :string]]]]]]])

(def ExportIdCollection
  [:or
   [:set [:or :string :keyword]]
   [:sequential [:or :string :keyword]]])

(def EndpointConfig
  (mu/closed-schema
   [:map
    [:on-error {:optional true} fn?]
    [:exports {:optional true}
     [:map
      [:events {:optional true} ExportIdCollection]
      [:subs {:optional true} ExportIdCollection]
      [:namespaces {:optional true} ExportIdCollection]]]
    [:transit-handlers {:optional true}
     [:map
      [:read {:optional true} [:map-of any? any?]]
      [:write {:optional true} [:map-of any? any?]]]]]))

(def QueryVector
  [:vector any?])

(defn message-schema-builder
  [body-schema]
  [:map
   [:body {:optional true} body-schema]
   [:headers {:optional true} [:map-of any? any?]]
   [:type [:enum :event :reply :handshake :handshake-reply]]
   [:request-id {:optional true} [:or :string :int]]])

(def Message
  [:multi {:dispatch :type}
   [:event (message-schema-builder [:vector any?])]
   [::m/default (message-schema-builder any?)]])
