;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.proxy
  (:refer-clojure :exclude [proxy send])
  (:require [fluxus.flow :as f]
            [fluxus.promise :as p]
            [tempus.duration :as td]
            [utilis.timer :as timer]
            [utilis.string :as ust]
            [spectator.log :as log]
            #?(:clj [clojure.pprint :refer [simple-dispatch]]))
  #?(:clj (:import [clojure.lang IObj IMeta])))

(declare pr-proxy)

(deftype Proxy [id token-fn encode decode connection
                handlers requests coeffects meta-map]
  Object
  (toString
    [^Proxy this]
    (pr-proxy this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash)
    [_]
    (hash [:via/service id]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv)
    [^Proxy this other]
    (boolean
     (when (instance? Proxy other)
       (= (#?(:clj .id :cljs .-id) this)
          (#?(:clj .id :cljs .-id) ^Proxy other)))))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Proxy. id token-fn encode decode connection
            handlers requests coeffects meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-proxy this)))]))

#?(:clj
   (defmethod print-method Proxy
     [^Proxy p w]
     (.write ^java.io.Writer w ^String (pr-proxy p))))

#?(:clj
   (defmethod simple-dispatch Proxy
     [^Proxy p]
     (print (pr-proxy p))))

(defn proxy
  [id token-fn encode decode connection-p]
  (Proxy. (atom id)
          token-fn
          encode
          decode
          (atom connection-p)
          (atom {:connect []
                 :disconnect []
                 :release []})
          (atom {})
          (atom nil)
          nil))

(defn on-connect
  [^Proxy proxy f]
  (let [connection (#?(:clj .connection :cljs .-connection) proxy)
        handlers (#?(:clj .handlers :cljs .-handlers) proxy)
        connection-p @connection]
    (when (and (p/realized? connection-p)
               (not (f/closed? @connection-p)))
      (f proxy))
    (swap! handlers update :connect conj f)))

(defn on-disconnect
  [^Proxy proxy f]
  (let [connection (#?(:clj .connection :cljs .-connection) proxy)
        handlers (#?(:clj .handlers :cljs .-handlers) proxy)
        connection-p @connection]
    (when (and (p/realized? connection-p)
               (f/closed? @connection-p))
      (f proxy))
    (swap! handlers update :disconnect conj f)))

(defn on-release
  [^Proxy proxy f]
  (let [handlers (#?(:clj .handlers :cljs .-handlers) proxy)]
    (swap! handlers update :release conj f)))

(defn token
  [^Proxy proxy]
  ((#?(:clj .token-fn :cljs .-token-fn) proxy)))

(defn send
  [^Proxy proxy message]
  (log/trace [:via/>? proxy message])
  (let [id (#?(:clj .id :cljs .-id) proxy)
        connection (#?(:clj .connection :cljs .-connection) proxy)
        encode (#?(:clj .encode :cljs .-encode) proxy)]
    (-> @connection
        (p/then
          (fn [connection]
            (log/trace [:via/>? @id connection message])
            (-> (f/put! connection (encode message))
                (p/then
                  (fn [v]
                    (log/trace [:via/> @id connection message v])))
                (p/catch
                  (fn [e]
                    (log/trace [:via/>! @id connection message e])
                    ;; TODO: Handle error
                    ))))))))

(defn invoke
  ([^Proxy proxy message]
   (invoke proxy message {}))
  ([^Proxy proxy message {:keys [timeout]
                          :or {timeout (td/seconds 60)}}]
   (log/trace [:via/><? proxy message timeout])
   (let [id (#?(:clj .id :cljs .-id) proxy)
         reply (p/promise {:label (str "(invoke " id " " (pr-str message) ")")})
         reply-id ((#?(:clj .token-fn :cljs .-token-fn) proxy))
         requests (#?(:clj .requests :cljs .-requests) proxy)]
     (try
       (swap! requests
              assoc reply-id
              {:on-reply
               (fn [value]
                 (when-not (p/realized? reply)
                   (p/resolve! reply value)))
               :timer
               (timer/run-after
                (fn []
                  (log/trace [:via/><! :timeout proxy message])
                  (when-not (p/realized? reply)
                    (p/reject! reply :via.invoke/timeout))
                  (swap! requests dissoc reply-id))
                (td/into :milliseconds timeout))})
       (send proxy (assoc message :reply-id reply-id))
       (catch #?(:clj Throwable :cljs :default) e
         (log/error [:via/><? proxy] e)))
     reply)))

(defn coeffects
  [^Proxy proxy]
  @(#?(:clj .coeffects :cljs .-coeffects) proxy))

;;; Private

(defn- pr-proxy
  [^Proxy p]
  (ust/format
   (str "#<via/proxy@" #?(:clj "0x%x" :cljs "%s") "%s%s>")
   (hash p)
   (str "[" @(#?(:clj .id :cljs .-id) p) "] ")
   (str (let [connection-p @(#?(:clj .connection :cljs .-connection) p)]
          (if (p/realized? connection-p)
            @connection-p
            "...")))
   (str ":requests " (vec (keys @(#?(:clj .requests :cljs .-requests) p))))))
