(ns via.core-test
  (:require [via.endpoint :as e]
            [via.subs :as vs]
            [via.events :as ve]
            [via.schema :as vsc]
            [signum.events :refer [reg-event]]
            [signum.subs :refer [reg-sub] :as sig-sub]
            [signum.signal :as sig]
            [vectio.http :as http]
            [vectio.netty.server :as ns]
            [tempus.core :as tempus]

            [clojure.java.io :as io]

            [spectator.log :refer [error info]]
            [clojure.test.check.generators :as gen]
            [clojure.test.check.properties :as prop]
            [clojure.test.check.clojure-test :refer [defspec]]
            [fluxus.flow :as f]
            [clojure.string :as st])
  (:import [io.netty.handler.ssl
            ApplicationProtocolConfig
            ApplicationProtocolConfig$Protocol
            ApplicationProtocolConfig$SelectorFailureBehavior
            ApplicationProtocolConfig$SelectedListenerFailureBehavior
            ApplicationProtocolNames
            SslContextBuilder
            SslProvider
            SupportedCipherSuiteFilter]
           [io.netty.handler.codec.http2 Http2SecurityUtil]
           [java.net ServerSocket]))

(f/set-default-on-error!
 (fn [stream error]
   (error :fluxus.flow/on-error
            {:stream stream
             :error error})))

(defn- gen-id
  []
  (str (java.util.UUID/randomUUID)))

(defn- allocate-free-port!
  []
  (let [socket (ServerSocket. 0)]
    (.setReuseAddress socket true)
    (let [port (.getLocalPort socket)]
      (try (.close socket) (catch Exception _))
      port)))

(defn endpoint
  ([] (endpoint nil))
  ([{:keys [on-close on-handshake on-handshake-reply] :as args}]
   (let [endpoint (e/init
                   (merge
                    {:on-error (fn [_])}
                    (dissoc args :on-close :on-handshake :on-handshake-reply)))
         ring-handler (fn [request]
                        (when-let [{:keys [stream response]} (ns/websocket-stream-response request)]
                          (f/on-close stream (fn [_] (when on-close (on-close))))
                          (e/register endpoint stream
                                      {:on-handshake on-handshake
                                       :on-handshake-reply on-handshake-reply})
                          response))
         port (allocate-free-port!)
         server (http/server
                 {:host "localhost"
                  :port port
                  :ring-handler ring-handler
                  :ssl-context (-> (SslContextBuilder/forServer
                                    (io/file "dev-resources/certs/server.crt")
                                    (io/file "dev-resources/certs/server.key"))
                                   (.sslProvider SslProvider/JDK)
                                   (.ciphers Http2SecurityUtil/CIPHERS SupportedCipherSuiteFilter/INSTANCE)
                                   (.applicationProtocolConfig
                                    (ApplicationProtocolConfig.
                                     ApplicationProtocolConfig$Protocol/ALPN
                                     ApplicationProtocolConfig$SelectorFailureBehavior/NO_ADVERTISE
                                     ApplicationProtocolConfig$SelectedListenerFailureBehavior/ACCEPT
                                     ^"[Ljava.lang.String;" (into-array [ApplicationProtocolNames/HTTP_2])))
                                   .build)})]
     {:endpoint endpoint
      :http-server server
      :port port})))

(defn wait-for
  ([p] (wait-for p 3000))
  ([p timeout-ms]
   (let [result (deref p timeout-ms ::timed-out)]
     (if (= result ::timed-out)
       (throw (ex-info "Timed out waiting for promise" {}))
       result))))

(defn connect
  ([e1 e2] (connect e1 e2 nil))
  ([{:keys [endpoint]}
    {:keys [port]}
    {:keys [on-close
            on-handshake
            on-handshake-reply
            reconnect
            on-client]}]
   (let [stream-fn #(let [client (http/websocket-client
                                  {:host "localhost"
                                   :port port
                                   :path "/"
                                   :ssl-context (unsafe-self-signed-ssl-context)})]
                      (when on-client (on-client @client))
                      client)]
     (if reconnect
       (let [connection (promise)]
         (e/reconnecting-stream
          endpoint
          (vus/reconnecting-stream stream-fn)
          {:on-connect (partial deliver connection)})
         (wait-for connection))
       (let [stream @(stream-fn)]
         (when on-close (f/on-close stream (fn [_] (on-close))))
         (e/register endpoint stream
                     {:on-handshake on-handshake
                      :on-handshake-reply on-handshake-reply}))))))

(defn shutdown
  [{:keys [http-server endpoint]}]
  (try (.close ^java.io.Closeable http-server)
       (catch Exception e
         (error e)))
  (try (e/halt endpoint)
       (catch Exception e
         (error e))))

(defspec test-send-directly-to-peer
  10
  (prop/for-all [value gen/any-printable-equatable]
                (let [event-id (str (gen-id) "/event")
                      e1 (endpoint)
                      e2 (endpoint {:exports {:events #{event-id}}})]
                  (reg-event
                   event-id
                   (fn [_ [_ value]]
                     {:via/reply {:status 200
                                  :body value}}))
                  (try (= value (:body @(ve/invoke
                                         (connect e1 e2)
                                         [event-id value]
                                         {:timeout 1000})))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))

(defspec test-send-dates-directly-to-peer
  10
  (prop/for-all [value (gen/map gen/keyword gen/string)]
                (let [value (assoc value :date-time (tempus/now))
                      event-id (str (gen-id) "/event")
                      e1 (endpoint)
                      e2 (endpoint {:exports {:events #{event-id}}})]
                  (reg-event
                   event-id
                   (fn [_ [_ value]]
                     {:via/reply {:status 200
                                  :body value}}))
                  (try (let [result @(ve/invoke
                                      (connect e1 e2)
                                      [event-id value]
                                      {:timeout 1000})]
                         (= (tempus/into :long (:date-time value))
                            (tempus/into :long (:date-time (:body result)))))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))

(defspec test-sub-updates-on-change
  10
  (prop/for-all [value (gen/sized (fn [size] (gen/vector gen/any-printable-equatable (max size 1))))]
                (let [sub-id (str (gen-id) "/sub")
                      e1 (endpoint)
                      e2 (endpoint {:exports {:subs #{sub-id}}})
                      value-signal (sig/signal ::init)
                      promises (mapv (fn [_] (promise)) value)]
                  (reg-sub sub-id (fn [_] @value-signal))
                  (try (let [sub (vs/subscribe (connect e1 e2) [sub-id])]
                         (add-watch sub ::watch (fn [_ _ _ {:keys [value i]}]
                                                  (when (number? i)
                                                    (deliver (nth promises i) value))))
                         (Thread/sleep 2000)
                         (doseq [[index value] (map-indexed vector value)]
                           (sig/alter! value-signal
                                       (fn [_]
                                         {:i index
                                          :value value}))
                           (wait-for (nth promises index)))
                         (let [result (mapv wait-for promises)]
                           (remove-watch sub ::watch)
                           (= result (vec value))))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))

(defspec test-subs-cleanup-properly
  10
  (prop/for-all [value gen/any-printable-equatable]
                (let [sub-id (str (gen-id) "/sub")
                      e1-promise (promise)
                      e1 (endpoint)
                      e2-promise (promise)
                      e2 (endpoint {:exports {:subs #{sub-id}}
                                    :on-close (fn [] (deliver e2-promise true))})
                      value-signal (sig/signal)]
                  (reg-sub sub-id (fn [_] @value-signal))
                  (try (let [sub (vs/subscribe
                                  (connect
                                   e1 e2
                                   {:on-close (fn [] (deliver e1-promise true))})
                                  [sub-id])]
                         (add-watch sub ::watch (fn [_ _ _ value]))
                         (sig/alter! value-signal (constantly value))
                         (remove-watch sub ::watch)
                         (shutdown e2)
                         (wait-for e1-promise)
                         (wait-for e2-promise)
                         ;; give fluxus an extra 1s to call close handlers
                         (Thread/sleep 1000)
                         (and (empty? @(:connections (:endpoint e2)))
                              (empty? @(:connections (:endpoint e1)))))
                       (catch Exception e
                         (error e)
                         (shutdown e2)
                         false)
                       (finally
                         (shutdown e1))))))

(defspec test-export-api-prevents-event-access
  10
  (prop/for-all [value gen/any-printable-equatable]
                (let [event-id (str (gen-id) "/event")
                      e1 (endpoint)
                      e2 (endpoint)]
                  (reg-event
                   event-id
                   (fn [_ [_ value]]
                     {:via/reply {:status 200
                                  :body value}}))
                  (try (= {:status 400}
                          (select-keys @(ve/invoke
                                         (connect e1 e2)
                                         [event-id value]
                                         {:timeout 1000})
                                       [:status]))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))

(defspec test-export-api-prevents-sub-access
  10
  (prop/for-all [value gen/any-printable-equatable]
                (let [sub-id (str (gen-id) "/sub")
                      e1 (endpoint)
                      e2 (endpoint)]
                  (reg-sub sub-id (fn [_] ::unauthorized))
                  (try (let [sub (vs/subscribe (connect e1 e2) [sub-id])]
                         (try (add-watch sub ::watch (fn [_ _ _ _]))
                              (catch Exception _))
                         (Thread/sleep 100)
                         (= (-> (try @sub
                                     (catch Exception e
                                       (ex-data e)))
                                :reply
                                :body
                                (select-keys [:status :error]))
                            {:status :error
                             :error :invalid-subscription}))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))

(defspec test-sub-reconnect-works
  10
  (prop/for-all [[value1 value2] (gen/vector-distinct gen/any-printable-equatable {:num-elements 2})]
                (let [sub-id (str (gen-id) "/sub")
                      disconnected-promise (promise)
                      e1 (endpoint)
                      e2 (endpoint {:exports {:subs #{sub-id}}})
                      value-signal (sig/signal ::init)
                      value-promise-1 (promise)
                      value-promise-2 (promise)
                      current-promise (atom value-promise-1)
                      initial-client (promise)
                      connection (connect e1 e2
                                          {:reconnect true
                                           :on-client #(when (not (realized? initial-client))
                                                         (f/on-close
                                                          % (fn [_]
                                                              (deliver disconnected-promise true)))
                                                         (deliver initial-client %))})]
                  (reg-sub sub-id (fn [_] @value-signal))
                  (wait-for initial-client)
                  (try (let [sub (vs/subscribe connection [sub-id])]
                         (add-watch sub ::watch (fn [_ _ _ value]
                                                  (when-let [p @current-promise]
                                                    (deliver p value))))
                         (sig/alter! value-signal (constantly value1))
                         (wait-for value-promise-1)
                         (Thread/sleep 500)
                         (f/close! @initial-client)
                         (wait-for disconnected-promise)
                         (Thread/sleep 500)
                         (reset! current-promise value-promise-2)
                         (sig/alter! value-signal (constantly value2))
                         (Thread/sleep 500)
                         (= @value-promise-2 value2))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))

(defspec test-handshake-transmits
  10
  (prop/for-all [value gen/any-printable-equatable]
                (let [e1-hs (promise)
                      e1-hs-reply (promise)
                      e1 (endpoint)
                      e2-hs (promise)
                      e2-hs-reply (promise)
                      e2 (endpoint {:on-handshake #(deliver e2-hs %)
                                    :on-handshake-reply #(deliver e2-hs-reply %)})]
                  (connect e1 e2 {:on-handshake #(deliver e1-hs %)
                                  :on-handshake-reply #(deliver e1-hs-reply %)})
                  (try (and (wait-for e1-hs)
                            (wait-for e1-hs-reply)
                            (wait-for e2-hs)
                            (wait-for e2-hs-reply))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))

(defspec test-send-large-message
  10
  (prop/for-all [_value gen/any-printable-equatable]
                (let [value (st/join "" (repeatedly 75000 #(rand-int 10)))
                      event-id (str (gen-id) "/event")
                      e1 (endpoint)
                      e2 (endpoint {:exports {:events #{event-id}}})]
                  (reg-event
                   event-id
                   (fn [_ [_ value]]
                     {:via/reply {:status 200
                                  :body value}}))
                  (try (= value (:body @(ve/invoke
                                         (connect e1 e2)
                                         [event-id value]
                                         {:timeout 1000})))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))


;;; Schema Tests

(defspec strip-inbound-unknown-keys
  10
  (prop/for-all [value (gen/map gen/keyword gen/any-printable-equatable)
                 known-value gen/keyword]
                (let [value (assoc value :known-value known-value)
                      event-id (str (gensym) "/event")
                      e1 (endpoint {:exports {:events #{event-id}}})
                      e2 (endpoint)]
                  (reg-event
                   event-id
                   (vsc/>fn
                       [_ [_ value]]
                     [_ [:tuple _ [:map [:known-value :keyword]]] => any?]
                     {:via/reply {:status 200
                                  :body value}}))
                  (try (= {:known-value known-value}
                          (:body @(ve/invoke
                                   (connect e2 e1)
                                   [event-id value])))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))

(defspec strip-outbound-unknown-keys
  10
  (prop/for-all [value (gen/map gen/keyword gen/any-printable-equatable)
                 known-value gen/keyword]
                (let [value (assoc value :known-value known-value)
                      event-id (str (gensym) "/event")
                      e1 (endpoint {:exports {:events #{event-id}}})
                      e2 (endpoint)]
                  (reg-event
                   event-id
                   (vsc/>fn
                       [_ [_ value]]
                     [_ [:tuple _ any?] => [:map [:known-value :keyword]]]
                     {:via/reply {:status 200
                                  :body value}}))
                  (try (= {:known-value known-value}
                          (:body @(ve/invoke
                                   (connect e2 e1)
                                   [event-id value])))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))

(defspec fail-bad-inbound-schema
  10
  (prop/for-all [value (gen/map gen/keyword gen/any-printable-equatable)]
                (let [event-id (str (gensym) "/event")
                      e1 (endpoint {:exports {:events #{event-id}}})
                      e2 (endpoint)]
                  (reg-event
                   event-id
                   (fn [& args]
                     (let [f (vsc/>fn
                                 [_ [_ value]]
                               [_ [:tuple _ [:map [:known-value :keyword]]] => any?]
                               {:via/reply {:status 200
                                            :body value}})]
                       (try (apply f args)
                            (catch Exception e
                              {:via/reply {:status 400
                                           :body {:error (if (re-find #"Arguments do not conform to schema" (.getMessage e))
                                                           :inbound-schema-validation-error
                                                           (do (println e)
                                                               :unknown-error))}}})))))
                  (try (let [reply (try @(ve/invoke
                                          (connect e2 e1)
                                          [event-id {:bad-value value}])
                                        (catch Exception e
                                          (:error (ex-data e))))]
                         (and (= :inbound-schema-validation-error (:error (:body reply)))
                              (= 400 (:status reply))))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))

(defspec fail-bad-outbound-schema
  10
  (prop/for-all [value (gen/map gen/keyword gen/any-printable-equatable)]
                (let [event-id (str (gensym) "/event")
                      e1 (endpoint {:exports {:events #{event-id}}})
                      e2 (endpoint)]
                  (reg-event
                   event-id
                   (fn [& args]
                     (let [f (vsc/>fn
                                 [_ [_ value]]
                               [_ [:tuple _ _] => [:map [:known-value :keyword]]]
                               {:via/reply {:status 200
                                            :body {:bad-value value}}})]
                       (try (apply f args)
                            (catch Exception e
                              {:via/reply {:status 500
                                           :body {:error (if (re-find #"Result does not conform to schema" (.getMessage e))
                                                           :outbound-schema-validation-error
                                                           (do (println e)
                                                               :unknown-error))}}})))))
                  (try (let [reply (try @(ve/invoke
                                          (connect e2 e1)
                                          [event-id value])
                                        (catch Exception e
                                          (:error (ex-data e))))]
                         (and (= :outbound-schema-validation-error (:error (:body reply)))
                              (= 500 (:status reply))))
                       (catch Exception e
                         (error e)
                         false)
                       (finally
                         (shutdown e1)
                         (shutdown e2))))))



(defn test-all
  []
  (info :test-send-directly-to-peer)
  (test-send-directly-to-peer 10)
  (info :test-send-dates-directly-to-peer)
  (test-send-dates-directly-to-peer 10)
  (info :test-sub-updates-on-change)
  (test-sub-updates-on-change 10)
  (info :test-subs-cleanup-properly)
  (test-subs-cleanup-properly 10)
  (info :test-export-api-prevents-event-access)
  (test-export-api-prevents-event-access 10)
  (info :test-export-api-prevents-sub-access)
  (test-export-api-prevents-sub-access 10)
  (info :test-handshake-transmits)
  (test-handshake-transmits 10)
  (info :test-sub-reconnect-works)
  (test-sub-reconnect-works 10)
  (info :test-send-large-message)
  (test-send-large-message 10)
  (info :strip-inbound-unknown-keys)
  (strip-inbound-unknown-keys 10)
  (info :strip-outbound-unknown-keys)
  (strip-outbound-unknown-keys 10)
  (info :fail-bad-inbound-schema)
  (fail-bad-inbound-schema 10)
  (info :fail-bad-outbound-schema)
  (fail-bad-outbound-schema 10))
