(ns via.schema
  (:require [malli.core :as m]
            [malli.error :as me]
            [malli.util :as mu]))

(defn validate-throw
  [schema x]
  (let [valid (m/validate schema x)]
    (when (not valid)
      (throw (ex-info "Schema Validation Error"
                      {:value x
                       :explanation (-> schema
                                        (m/explain x)
                                        me/with-spell-checking
                                        me/humanize)})))
    x))

(def Some
  [:fn {:error/fn (fn [{:keys [value]} _]
                    "Value must not be nil.")}
   some?])

(def some-validator (m/validator Some))
(defn validate-some
  [& args]
  (when (not (every? some-validator args))
    (throw (ex-info "Schema Validation Error"
                    {:explanation {:message "Arguments must not be nil."
                                   :args args}}))))

(def ExportType
  [:enum :sub :event :namespace])

(def Address
  [:or
   :string
   [:map
    [:host :string]
    [:port int?]
    [:scheme :string]
    [:path :string]
    [:tls [:map
           [:server [:map
                     [:cert :string]
                     [:key :string]]]
           [:client [:map
                     [:ca :string]]]]]]])

(def ExportIdCollection
  [:or
   [:set [:or :string :keyword]]
   [:sequential [:or :string :keyword]]])

(def EndpointConfig
  (mu/closed-schema
   [:map
    [:on-error {:optional true} fn?]
    [:streams {:optional true}
     [:or
      [:set any?]
      [:sequential any?]]]
    [:always-send-whole-values {:optional true} :boolean]
    [:allow-deferred-subs {:optional true} :boolean]
    [:exports {:optional true}
     [:map
      [:events {:optional true} ExportIdCollection]
      [:subs {:optional true} ExportIdCollection]
      [:namespaces {:optional true} ExportIdCollection]]]
    [:transit-handlers {:optional true}
     [:map
      [:read {:optional true} [:map-of any? any?]]
      [:write {:optional true} [:map-of any? any?]]]]]))

(def QueryVector
  [:vector any?])

(defn message-schema-builder
  [body-schema]
  [:map
   [:body {:optional true} body-schema]
   [:headers {:optional true} [:map-of any? any?]]
   [:type [:enum :event :reply :handshake :handshake-reply]]
   [:request-id {:optional true} [:or :string :int]]])

(def Message
  [:multi {:dispatch :type}
   [:event (message-schema-builder [:vector any?])]
   [::m/default (message-schema-builder any?)]])
