;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.subs
  (:require [via.endpoint :as e]
            [via.events :as ve]
            [via.schema :as schema]
            [via.util.id :as id]
            [tempus.core :as t]
            [distantia.core :refer [patch diff]]
            [signum.subs :as ss]
            [signum.events :as se]
            [signum.signal :refer [signal alter!]]
            #?(:cljs [utilis.js :as j])
            #?(:clj [manifold.deferred :as d]
               :cljs [vectio.util.manifold.deferred :as d])
            [integrant.core :as ig]))

(declare local-query subscription-lock reg-sub)

(defn subscribe
  ([connection query]
   (subscribe connection query nil))
  ([connection query
    {:keys [init-timeout]
     :or {init-timeout 30000}}]
   (schema/validate-some connection query)
   (let [[local-query-id & _ :as local-query] (local-query connection query)
         {:keys [endpoint]} connection
         endpoint (endpoint)]
     (locking subscription-lock
       (when (not (ss/sub? local-query-id))
         (reg-sub endpoint connection
                  {:local-query local-query
                   :remote-query-id (first query)
                   :timeout init-timeout})))
     (ss/subscribe local-query))))

(defn init
  [endpoint]
  (e/export endpoint :event :-vs/s)
  (e/export endpoint :event :-vs/d)
  (e/export endpoint :event :-vss/u)
  endpoint)

(defmethod ig/init-key :via/subs
  [_ {:keys [endpoint]}]
  (init endpoint))


;;; Private

(defonce ^:private subscription-lock
  #?(:clj (Object.)
     :cljs (js/Object.)))

(defn- local-query
  [connection query]
  (let [local-query-id [(:id connection) (first query)]]
    (cons local-query-id (rest query))))

(defn- register-state
  [{:keys [next-id subs]} state]
  (let [signal-id (next-id)]
    (swap! (:signals subs) assoc signal-id state)
    signal-id))

(defn- deregister-state
  [{:keys [subs]} signal-id]
  (swap! (:signals subs) dissoc signal-id)
  signal-id)

(defn- state
  [{:keys [subs]} signal-id]
  (get @(:signals subs) signal-id))

(defn- reg-sub
  [endpoint connection {:keys [local-query remote-query-id timeout]}]
  (let [[local-query-id & _] local-query
        connection-id (atom (:id connection))
        connection* (fn [] (get @(:connections endpoint) @connection-id))]
    (ss/reg-sub
     local-query-id
     (fn [query-v]
       (let [signal (signal)
             changes (atom nil)
             remote-query-v (vec (cons remote-query-id (rest query-v)))
             remote-subscribe (fn [signal-id]
                                (reset! changes {:window [] :sn 0})
                                (-> (connection*)
                                    (ve/invoke
                                     [:-vs/s
                                      {:q remote-query-v
                                       :c [:-vss/u signal-id]}]
                                     {:timeout timeout})
                                    (d/on-realized
                                     (fn [{:keys [status] :as reply}]
                                       (when (not= status 200)
                                         (when-let [connection (connection*)]
                                           (let [message "An error occurred initiating a remote subscription"
                                                 data {:reply reply
                                                       :query-v remote-query-v}]
                                             (deregister-state connection signal-id)
                                             (e/handle-error
                                              endpoint :via.error/sub-init-error
                                              message data)
                                             (->> (ex-info message data)
                                                  constantly
                                                  (alter! signal))))))
                                     (fn [error]
                                       (when-let [connection (connection*)]
                                         (let [message "An error occurred initiating a remote subscription"
                                               data {:error error
                                                     :query-v remote-query-v}]
                                           (deregister-state connection signal-id)
                                           (e/handle-error
                                            endpoint :via.error/sub-init-error
                                            message data)
                                           (alter! signal
                                                   (constantly
                                                    (ex-info
                                                     "An error occurred initiating a remote subscription"
                                                     {:error error
                                                      :query-v remote-query-v})))))))))
             signal-id (register-state
                        connection {:signal signal
                                    :changes changes
                                    :remote-subscribe remote-subscribe
                                    :query-v query-v
                                    :remote-query-v remote-query-v})]
         (if @(:ready connection)
           (remote-subscribe signal-id)
           (add-watch (:ready connection) signal-id
                      (fn [_ _ old-ready? new-ready?]
                        (when (and (not old-ready?) new-ready?)
                          (remote-subscribe signal-id)))))
         [signal-id remote-query-v signal]))
     (fn [[signal-id query-v _] _]
       (let [connection (connection*)]
         (remove-watch (:ready connection) signal-id)
         (deregister-state connection signal-id)
         (ve/dispatch connection [:-vs/d {:query-v query-v}])))
     (fn [[_ _ signal] _] @signal))))

(defn- register-cleanup-fn
  [{:keys [subs] :as connection} query-v cleanup]
  (when-let [cleanup (get-in @(:remote-dispose subs)
                             [(:id connection) query-v])]
    (cleanup))
  (swap! (:remote-dispose subs)
         assoc-in [(:id connection) query-v]
         cleanup))

(defn- cleanup
  [{:keys [subs] :as connection} query-v]
  (boolean
   (when-let [cleanup (get-in @(:remote-dispose subs)
                              [(:id connection) query-v])]
     (swap! (:remote-dispose subs)
            #(if-let [cleanup-fns (-> %
                                      (get (:id connection))
                                      (dissoc query-v)
                                      not-empty)]
               (assoc % (:id connection) cleanup-fns)
               (dissoc % (:id connection))))
     (cleanup)
     true)))

(defn- handle-subscribe
  [{:keys [endpoint connection]} [_ {:keys [q c]}]]
  (let [query-v q
        callback c]
    (boolean
     (let [[query-id & _] query-v]
       (locking subscription-lock
         (when (and (e/exported? endpoint :sub query-id)
                    (ss/sub? query-id))
           (when-let [signal (binding [ss/*context* {:coeffects
                                                     (merge {:endpoint endpoint
                                                             :connection connection}
                                                            (:coeffects connection))}]
                               (ss/subscribe query-v))]
             (let [watch-key (id/uuid)
                   sn (atom -1)
                   cleanup #(remove-watch signal watch-key)]
               (register-cleanup-fn connection query-v cleanup)
               (add-watch signal watch-key
                          (fn [_ _ old new]
                            (ve/dispatch
                             connection
                             (conj (vec callback)
                                   (if (instance? #?(:clj java.lang.Throwable
                                                     :cljs js/Error)
                                                  new)
                                     {:c [:v :signum.signal/cold]
                                      :sn (swap! sn inc)
                                      :error (let [ex-data (ex-data new)]
                                               (cond-> {:message #?(:clj (.getMessage ^java.lang.Throwable new)
                                                                    :cljs (j/get new :message))}
                                                 ex-data (assoc :data (try (e/encode* endpoint ex-data)
                                                                           ex-data
                                                                           (catch #?(:cljs js/Error
                                                                                     :clj Exception) _e
                                                                             (pr-str ex-data))))))}
                                     {:c (if (and (not (:always-send-whole-values endpoint))
                                                  (not= old new)
                                                  (or (and (map? new) (map? old))
                                                      (and (vector? new) (vector? old))))
                                           [:p (diff old new)]
                                           [:v new])
                                      :sn (swap! sn inc)}))))))
             true)))))))

(defn- split-contiguous
  [last-sn window]
  (let [window (sort-by :sn window)
        state (volatile! last-sn)
        result (mapv vec (partition-by #(= (:sn %) (vswap! state inc)) window))]
    (cond
      (= (count result) 2) result
      (= (inc last-sn) (:sn (ffirst result))) [(first result) nil]
      :else [nil (first result)])))

(defn- write-message!
  [signal {:keys [c]}]
  (alter! signal
          (fn [value]
            (if (= :v (first c))
              (second c)
              (patch value (second c))))))

(defn- printable-connection
  [connection]
  (dissoc connection
          :stream
          :sink
          :on-deregister
          :endpoint
          :subs
          :requests
          :coeffects))

;;; Event Handlers

(se/reg-event
 :-vs/s
 (fn [{:keys [endpoint connection] :as context} query-v]
   (if (handle-subscribe context query-v)
     {:via/reply {:body {:status :success}
                  :status 200}}
     (do (e/handle-error
          endpoint
          :via.error/unknown-or-non-exported-sub
          "A remote peer attempted to subscribe to an unknown or non-exported sub"
          {:query-v query-v
           :connection (printable-connection connection)})
         {:via/reply {:status 400
                      :body {:status :error
                             :error :invalid-subscription
                             :query-v query-v}}}))))

(se/reg-event
 :-vs/d
 (fn [{:keys [connection]} [_ {:keys [query-v]}]]
   (if (cleanup connection query-v)
     {:via/reply {:status 200}}
     {:via/reply {:status 400
                  :body {:error :invalid-subscription
                         :query-v query-v}}})))

(se/reg-event
 :-vss/u
 (fn [{:keys [endpoint connection]} [_ signal-id message & _ :as query-v]]
   (when-let [error (:error message)]
     (e/handle-error
      endpoint
      :via.error.subs/remote-error
      "A remote subscription is in an error state"
      (merge {:error error
              :query-v (:remote-query-v (state connection signal-id))}
             #?(:clj {:connection (printable-connection connection)}))))
   (if-let [{:keys [signal changes]} (state connection signal-id)]
     (let [{:keys [window last-sn] :or {last-sn -1}} @changes
           [contiguous-messages window] (split-contiguous last-sn (conj window message))]
       (reduce write-message! signal contiguous-messages)
       (reset! changes {:window window
                        :updated (t/into :long (t/now))
                        :last-sn (-> contiguous-messages last :sn)})
       {:via/reply {:status 200}})
     (do (e/handle-error
          endpoint
          :via.error/unknown-signal-id
          "A remote peer attempted to set the value of a subscription using an unknown signal-id"
          {:query-v query-v
           :connection (printable-connection connection)})
         {:via/reply {:status 400
                      :body {:error :unknown-signal-id
                             :query-v query-v}}}))))
