;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.netty
  (:require [via.netty.http2.server :as h2]
            [via.netty.http.server :as http1]
            [utilis.map :refer [map-vals compact]]
            [manifold.executor :as e]
            [clojure.string :as st])
  (:import [io.netty.bootstrap ServerBootstrap]
           [io.netty.channel
            ChannelDuplexHandler
            ChannelOption
            EventLoopGroup
            ChannelHandlerContext
            ChannelInitializer]
           [java.util.concurrent
            ThreadFactory
            Executors
            ExecutorService]
           [io.netty.handler.logging LoggingHandler LogLevel]
           [io.netty.handler.codec DecoderException]
           [io.netty.handler.codec.http.websocketx.extensions WebSocketExtensionData]
           [io.netty.handler.codec.http.websocketx.extensions.compression
            PerMessageDeflateServerExtensionHandshaker]
           [io.netty.channel.nio NioEventLoopGroup]
           [io.netty.channel.socket
            ServerSocketChannel
            SocketChannel]
           [io.netty.channel.socket.nio NioServerSocketChannel]
           [io.netty.handler.ssl
            SslContext
            ApplicationProtocolNegotiationHandler
            ApplicationProtocolNames
            NotSslRecordException]))

(declare enumerating-thread-factory init-channel)

(defn start-server
  [{:keys [ssl-context socket-address handler]}]
  (when (not ssl-context) (throw (Exception. "SSL Context Required")))
  (let [num-cores      (.availableProcessors (Runtime/getRuntime))
        num-threads    (* 2 num-cores)
        thread-factory (enumerating-thread-factory "via-netty-http2-server-event-pool" false)
        closed?        (atom false)
        transport      :nio
        ^Class channel NioServerSocketChannel
        ^EventLoopGroup group (NioEventLoopGroup. num-threads thread-factory)
        handler-exec-service (Executors/newFixedThreadPool num-threads)]
    (try
      (let [b (doto (ServerBootstrap.)
                (.option ChannelOption/SO_BACKLOG (int 1024))
                (.option ChannelOption/SO_REUSEADDR true)
                (.option ChannelOption/MAX_MESSAGES_PER_READ Integer/MAX_VALUE)
                (.group group)
                (.channel channel)
                (.childHandler (proxy [ChannelInitializer] []
                                 (initChannel [^SocketChannel ch]
                                   (init-channel ssl-context handler-exec-service ch handler))))
                (.childOption ChannelOption/SO_REUSEADDR true)
                (.childOption ChannelOption/MAX_MESSAGES_PER_READ Integer/MAX_VALUE))
            ^ServerSocketChannel ch (-> b (.bind socket-address) .sync .channel)]
        (reify
          java.io.Closeable
          (close [_]
            (when (compare-and-set! closed? false true)
              (-> ch .close .sync)
              (-> group .shutdownGracefully)
              (.shutdown handler-exec-service)))
          Object
          (toString [_]
            (format "ViaHttp2Server[channel:%s, transport:%s]" ch transport))))
      (catch Exception e
        @(.shutdownGracefully group)
        (.shutdown handler-exec-service)
        (throw e)))))

(defn websocket-request?
  [req]
  (boolean
   (condp = (:protocol-version req)
     "http/2" (h2/websocket-request? req)

     "http/1.1" (http1/websocket-request? req)

     false)))

(defn websocket-accept-response
  [req handlers]
  (condp = (:protocol-version req)
    "http/2"
    (let [version (get-in req [:headers "sec-websocket-version"])
          subprotocols (get-in req [:headers "sec-websocket-protocol"])
          extensions (when-let [extensions (not-empty (get-in req [:headers "sec-websocket-extensions"]))]
                       (->> (st/split extensions #",")
                            (map #(st/split % #";"))
                            (into {})
                            (map-vals #(->> (st/split (st/trim %) #";")
                                            (map (fn [parameter]
                                                   (let [[k v] (st/split parameter #"\=")]
                                                     [k v])))
                                            (into {})))))
          compression-extension (when-let [parameters (get extensions "permessage-deflate")]
                                  (let [compression-handshaker (PerMessageDeflateServerExtensionHandshaker.)]
                                    (.handshakeExtension compression-handshaker
                                                         (WebSocketExtensionData.
                                                          "permessage-deflate"
                                                          parameters))))]
      (with-meta (compact
                  {:status 200
                   :headers (when compression-extension
                              (let [response-data (.newReponseData compression-extension)
                                    param-string (when-let [param-string (->> (.parameters response-data)
                                                                              (map (fn [[k v]] (str k "=" v)))
                                                                              (st/join "; ")
                                                                              not-empty)]
                                                   (str "; " param-string))]
                                {"sec-websocket-extensions" (str "permessage-deflate" param-string)}))})
        {:compression-extension compression-extension
         :handlers handlers}))

    "http/1.1"
    (with-meta {:status 101}
      {:handlers handlers})

    nil))


;;; Private

(defn apn-handler
  [^ExecutorService exec-service handler]
  (proxy [ApplicationProtocolNegotiationHandler] [ApplicationProtocolNames/HTTP_1_1]
    (configurePipeline [^ChannelHandlerContext ctx ^String protocol]
      (cond
        (.equals ApplicationProtocolNames/HTTP_2 protocol)
        (h2/configure-http2-pipeline (.pipeline ctx) exec-service handler)

        (.equals ApplicationProtocolNames/HTTP_1_1 protocol)
        (http1/configure-http1-pipeline (.pipeline ctx) exec-service handler)

        :else (throw (IllegalStateException. (str "Protocol: " protocol " not supported.")))))))

(defn ssl-exception-handler
  []
  (proxy [ChannelDuplexHandler] []
    (exceptionCaught [^ChannelHandlerContext ctx ^Throwable cause]
      (if (and (instance? DecoderException cause)
               (instance? NotSslRecordException (.getCause cause)))
        (.close ctx)
        (throw cause)))))

(defn- init-channel
  [^SslContext ssl-context ^ExecutorService exec-service ^SocketChannel ch ^clojure.lang.IFn handler]
  (doto (.pipeline ch)
    (.addLast "ssl-handler" (.newHandler ssl-context (.alloc ch)))
    (.addLast "ssl-exception-handler" (ssl-exception-handler))
    (.addLast "apn-handler" (apn-handler exec-service handler))))

(defn ^ThreadFactory enumerating-thread-factory
  [prefix daemon?]
  (let [num-threads (atom 0)]
    (e/thread-factory
     #(str prefix "-" (swap! num-threads inc))
     (deliver (promise) nil)
     nil
     daemon?)))
