;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.adapters.netty
  (:refer-clojure :exclude [subs send])
  (:require [via.adapter :as adapter]
            [via.netty :as netty]
            [via.util.request :refer [parse-query-string]]
            [via.util.id :refer [uuid]]))

(declare send* disconnect* handle-request)

(defn adapter
  [{:keys [client-connect handle-connection] :as opts}]
  (let [endpoint (reify adapter/Endpoint
                   (opts [endpoint] opts)
                   (send [endpoint peer-id message]
                     (send* endpoint peer-id message))
                   (disconnect [endpoint peer-id]
                     (disconnect* endpoint peer-id))
                   (connect [endpoint address]
                     (client-connect endpoint address))
                   (shutdown [endpoint]
                     ))]
    (when handle-connection
      (adapter/add-event-listener endpoint :via.endpoint.peer/connected (partial handle-connection endpoint)))
    (fn ([] endpoint)
      ([request]
       (handle-request endpoint request)))))

;;; Private

(defn- send*
  [endpoint peer-id message]
  (boolean
   (when message
     (when-let [{:keys [send close]} (get-in @(adapter/peers endpoint) [peer-id :connection])]
       (send message)
       true))))

(defn- disconnect*
  [endpoint peer-id]
  (when-let [{:keys [close]} (get-in @(adapter/peers endpoint) [peer-id :connection])]
    (close)))

(defn- handle-request
  [endpoint request]
  (when (netty/websocket-request? request)
    (netty/websocket-accept-response
     request
     (let [request (cond-> request
                     (not (:query-params request)) (assoc :query-params (parse-query-string (:query-string request))))
           peer-id (or (get-in request [:headers "sec-websocket-key"])
                       (uuid))]
       {:on-open (fn [connection]
                   ((adapter/handle-connect endpoint) (constantly endpoint)
                    {:id peer-id
                     :connection connection
                     :role :terminator
                     :params (:query-params request)
                     :request (assoc request :peer-id peer-id)}))
        :on-close (fn []
                    ((adapter/handle-disconnect endpoint)
                     (constantly endpoint)
                     (get @(adapter/peers endpoint) peer-id)))
        :on-text-message (fn [message]
                           ((adapter/handle-message endpoint)
                            (constantly endpoint)
                            (assoc request :peer-id peer-id)
                            message))}))))
